using System;
using System.Collections;
using System.Reflection;
using System.Drawing;

using Microsoft.DirectX;
using Microsoft.DirectX.DirectInput;

using DarkStrideToolbox;


namespace DarkStride.StellarLanes.SharedDLL
{
	public class Module : Item
	{
		#region Properties
		private Window_Popup m_oPopup = null;
		private Chassis m_oChassis = null;
		private int m_nXOffset = 0;
		private int m_nYOffset = 0;
		private int m_nMGridWidth = 0;
		private int m_nMGridHeight = 0;
		private enumChassisDirection m_nDirection = enumChassisDirection.Up;
		private string m_sTexture_Editor = "";
		private string m_sTexture_Space = "";
		#endregion

		
		public Module()
		{
			this.PKeyObject = Globals.Inst().PrimaryKey.GetNewPrimarykey();
		}
		public Module( ComplexEntity oParentEntity )
		{
			this.PKeyObject = Globals.Inst().PrimaryKey.GetNewPrimarykey();
			//m_sGUID = DSMisc.GetGUID();
			this.ParentEntity = oParentEntity;
		}

		public override DSSerialize Serialize(Session oSession)
		{
			//ItemSpecialTrait oLoopItem = null;
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
			oSerialize.Set( 0,m_nXOffset );
			oSerialize.Set( 1,m_nYOffset );
			oSerialize.Set( 2,this.PKeyObject.Serialize() );
			//oSerialize.Set( 3,base.Serialize() );
            oSerialize.Set(3, base.Serialize(oSession));
			

			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			//ItemSpecialTrait oLoopItem = null;
			//DSSerialize oSerItem = null;
			//ArrayList oItemList = null;

			m_nXOffset = oSerialize.GetInt( 0 );
			m_nYOffset = oSerialize.GetInt( 1 );
			this.PKeyObject.DeSerialize( (DSSerialize)oSerialize.Get( 2 ) );
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 3 ) );
		}

		public virtual void PreRenderRenderSettingsChange( RenderSettings oRenderSettings )
		{
		}
		public virtual void Render( RenderSettings oRenderSettings,Chassis oParentChassis )
		{
			if( oRenderSettings.RenderType == enumRenderType.Modules && oRenderSettings.InGameScreen == enumGameScreen.Editor && 
				m_sTexture_Editor.Length > 0 )
			{
				RenderModule3D( m_sTexture_Editor,oRenderSettings,0 );
			}
			else if( oRenderSettings.RenderType == enumRenderType.Modules && oRenderSettings.InGameScreen != enumGameScreen.Editor && 
				m_sTexture_Space.Length > 0 )
			{
				RenderModule3D( m_sTexture_Space,oRenderSettings,this.ParentEntity.Angle );
			}
		}
		public override void RenderForEditor3D( System.Drawing.Rectangle oAreaToDrawAt,RenderSettings oRenderSettings )
		{
			if( oRenderSettings.RenderType == enumRenderType.Modules && 
				oRenderSettings.InGameScreen == enumGameScreen.Editor && 
				(
					( 
						m_sTexture_Editor.Length > 0 &&
						(
							oRenderSettings.EditorDisplay == enumEditorObject.Modules ||
							oRenderSettings.EditorDisplay == enumEditorObject.Unassigned 
						)
					)
					||
					(	 
						m_sTexture_Space.Length > 0 &&
						oRenderSettings.EditorDisplay == enumEditorObject.Skins 
					)
				)
			  )
			{
				Globals.Inst().GameEngine.RenderTexture2D( 
					m_sTexture_Editor,
					System.Drawing.Rectangle.Empty,oAreaToDrawAt,Vector2.Empty,
					0,oRenderSettings.PercentTransparent,false,oRenderSettings.BaseDrawColor.ToArgb() );
			}
		}
		public void RenderModule3D( string sGraphicKey,RenderSettings oRenderSettings,double nAngle )
		{
			Vector2 vRotatedScreenPtOfModule = Vector2.Empty;
			Vector2 vRotatedWorldPtOfModule = Vector2.Empty;
			Vector2 vScreenPtCenterShip = Vector2.Empty;
			Vector2 vWorldPtUpperLeftModuleCorner = Vector2.Empty;
			Vector2 vScreenPtUpperLeftModuleCorner = Vector2.Empty;
			Vector2 vRotate = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			if( oRenderSettings.EntityToCenterOn == this.ParentEntity )
			{
				//This Module upper left corner location
				vWorldPtUpperLeftModuleCorner = GetUpperLeftCornerWorldPt();
				vScreenPtUpperLeftModuleCorner = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vWorldPtUpperLeftModuleCorner );
				//Get the center of my ship
				vScreenPtCenterShip = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,this.ParentEntity.Pos );
				//Center of the module, rotated
				vRotatedWorldPtOfModule = GetRotatedCenterWorldPt();
				vRotatedScreenPtOfModule = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vRotatedWorldPtOfModule );
				oRenderRect = new System.Drawing.Rectangle( 
					(int)( vRotatedScreenPtOfModule.X - ( m_nMGridWidth * GraphicConstants.m_cGRID_WIDTH * oRenderSettings.ZoomLevel ) / 2.0 ),
					(int)( vRotatedScreenPtOfModule.Y - ( m_nMGridHeight * GraphicConstants.m_cGRID_HEIGHT * oRenderSettings.ZoomLevel ) / 2.0 ),
					(int)( m_nMGridWidth * GraphicConstants.m_cGRID_WIDTH * oRenderSettings.ZoomLevel ),
					(int)( m_nMGridHeight * GraphicConstants.m_cGRID_HEIGHT * oRenderSettings.ZoomLevel ) );

				oRenderRect = new System.Drawing.Rectangle(
					(int)vScreenPtUpperLeftModuleCorner.X,
					(int)vScreenPtUpperLeftModuleCorner.Y,
					(int)( m_nMGridWidth * GraphicConstants.m_cGRID_WIDTH * oRenderSettings.ZoomLevel ),
					(int)( m_nMGridHeight * GraphicConstants.m_cGRID_HEIGHT * oRenderSettings.ZoomLevel ) );

				//Where do we rotate around?  The center of the ship's difference from our upper left corner
                if (oRenderRect.Width > 0 && oRenderRect.Height > 0)
                {
                    vRotate = new Vector2((float)(m_nMGridWidth * GraphicConstants.m_cGRID_WIDTH * oRenderSettings.ZoomLevel) / 2.0f,
                        (float)(m_nMGridHeight * GraphicConstants.m_cGRID_HEIGHT * oRenderSettings.ZoomLevel) / 2.0f);

                    Vector2 vModuleWorldPtCenter = new Vector2(
                        vWorldPtUpperLeftModuleCorner.X + (m_nMGridWidth / 2.0f) * GraphicConstants.m_cGRID_WIDTH,
                        vWorldPtUpperLeftModuleCorner.Y + (m_nMGridHeight / 2.0f) * GraphicConstants.m_cGRID_HEIGHT);
                    Vector2 vScreenPtCenterModule = Globals.Inst().Session.ConvertWorldPtToScreenPt(oRenderSettings, vModuleWorldPtCenter);
                    vRotate = new Vector2(
                        vScreenPtCenterShip.X - vScreenPtUpperLeftModuleCorner.X,
                        vScreenPtCenterShip.Y - vScreenPtUpperLeftModuleCorner.Y);

                    //Now, render it.
                    Globals.Inst().GameEngine.RenderTexture2D(sGraphicKey, System.Drawing.Rectangle.Empty,
                        oRenderRect, vRotate, nAngle, oRenderSettings.PercentTransparent, false,
                        oRenderSettings.BaseDrawColor.ToArgb());
                }
			}
		}

		public override void HoverStart()
		{
			m_oPopup = new Window_Popup();
			m_oPopup.Display( Globals.Inst().GameEngine.MouseCursor,
				this,m_sTexture_Editor,this.GetHoverRenderText() );
		}
		public override void HoverStop()
		{
			if( m_oPopup != null )
			{
				m_oPopup.Dispose();
				m_oPopup = null;
			}
		}
		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();
		
			//Build the string
			oProperties.Add( "Structure: " + this.StructurePoints.ToString( "0" ) + " / " + this.MaxStructurePoints.ToString( "0" ) );

			return( oProperties );
		}

		public virtual Vector2 GetUpperLeftCornerWorldPt()
		{
			Vector2 vWorldPtUpperLeftChassisCorner = Vector2.Empty;
			Vector2 vWorldPtUpperLeftModuleCorner = Vector2.Empty;


			//The center point is actually the center of the base Module
			vWorldPtUpperLeftChassisCorner = m_oChassis.GetUpperLeftCornerWorldPt();
			//Now convert that based on our offset
			vWorldPtUpperLeftModuleCorner = new Vector2( 
				vWorldPtUpperLeftChassisCorner.X + m_nXOffset * GraphicConstants.m_cGRID_WIDTH,
				vWorldPtUpperLeftChassisCorner.Y - m_nYOffset * GraphicConstants.m_cGRID_HEIGHT );


			return( vWorldPtUpperLeftModuleCorner );
		}
		public virtual Vector2 GetRotatedCenterWorldPt()
		{
			Vector2 vModuleWorldPtUpperLeftCorner = Vector2.Empty;
			Vector2 vCenterWorldPt = Vector2.Empty;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;


			vModuleWorldPtUpperLeftCorner = GetUpperLeftCornerWorldPt();
			vCenterWorldPt = new Vector2( 
				(float)( vModuleWorldPtUpperLeftCorner.X + ( m_nMGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
				(float)( vModuleWorldPtUpperLeftCorner.Y - ( m_nMGridHeight / 2.0 ) * GraphicConstants.m_cGRID_HEIGHT ) );
			vRotatedLaunchWorldPt = this.ParentEntity.GetRotatedWorldPt( vCenterWorldPt );


			return( vRotatedLaunchWorldPt );
		}
		public virtual bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			enumModuleType nTypeOfSquare = MGridTypeNeeded( nXOffset,nYOffset ); 

			if( ( (int)nTypeOfSquare & (int)enumModuleType.OpenSpace ) == (int)enumModuleType.OpenSpace )
			{
				return( false );
			}
			else
			{
				return( true );
			}
		}
		public virtual enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			return( enumModuleType.All );
		}

		public virtual void Assign()
		{
		}
		public virtual void UnAssign()
		{
		}
		internal bool ModuleIsAvailable()
		{
			return( this.ParentEntity != null && this.ParentEntity.IsDead == false && 
					this.StructurePoints > 0 );
		}
        internal bool CanLaunchProjectile()
        {
            return ( this.ParentEntity.OwnerSocketID == MiscConstants.m_cNOSOCKETIDASSIGNED ||
                       (
                            Globals.Inst().OurShip != null &&
                            this.ParentEntity.OwnerSocketID == Globals.Inst().OurShip.OwnerSocketID
                       )
                   );
        }

        public virtual void KeyProcess(Session oSession, Microsoft.DirectX.DirectInput.Key oKey, bool bPressed)
		{
		}
        public virtual void NetworkMessageReceived(Session oSession, long nMessageType, string sMessage)
		{
		}
		public virtual double DealDamage( double nAmount,Region oDamageWhere )
		{
			double nDamageLeftOver = 0;
			double nFixedRedux = 0;
			double nPercRedux = 0;
			double nNewAmount = 0;


			nFixedRedux = this.GetPropertyTotal( enumEntProperties.Structure_FixedDamageRedux,false );
			nPercRedux = this.GetPropertyTotal( enumEntProperties.Structure_FixedDamageRedux,false );
			nNewAmount = base.ReduceDamage( nAmount,nFixedRedux,nPercRedux );

			if( nNewAmount > this.StructurePoints )
			{
				nDamageLeftOver = nNewAmount - this.StructurePoints;
				this.StructurePoints = 0;
			}
			else
			{
				this.StructurePoints -= nNewAmount;
				base.StructurePoints = DSMisc.Max( 0,base.StructurePoints );
			}
			return( nDamageLeftOver );
		}
		public virtual double GetPreArmorDamageReduction( double nDamage )
		{
			return( nDamage );
		}
		public void ApplyThrust( double nThrustAmount )
		{
			Vector2 vOffsetFromCOM = GetUpperLeftCornerWorldPt() - this.ParentEntity.Pos + 
				new Vector2( ( m_nMGridWidth * GraphicConstants.m_cGRID_WIDTH ) / 2.0f,
				( m_nMGridHeight * GraphicConstants.m_cGRID_HEIGHT ) / 2.0f );
			this.ParentEntity.ApplyThrust( vOffsetFromCOM,this.ParentEntity.Angle,nThrustAmount );
		}


		#region Properties
		public int XOffset
		{
			get
			{
				return( m_nXOffset );
			}
			set
			{
				m_nXOffset = value;
			}
		}
		public int YOffset
		{
			get
			{
				return( m_nYOffset );
			}
			set
			{
				m_nYOffset = value;
			}
		}
		public virtual enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.All );
			}
		}
		public virtual int MGridWidth 
		{
			get
			{
				return( m_nMGridWidth );
			}
			set
			{
				m_nMGridWidth = value;
			}
		}
		public virtual int MGridHeight
		{
			get
			{
				return( m_nMGridHeight );
			}
			set
			{
				m_nMGridHeight = value;
			}
		}
		public virtual enumChassisDirection Direction
		{
			get
			{
				return( m_nDirection );
			}
			set
			{
				m_nDirection = value;
			}
		}
		public Chassis Chassis
		{
			get
			{
				return( m_oChassis );
			}
			set
			{
				m_oChassis = value;
			}
		}
		public string Texture_Editor
		{
			get
			{
				return( m_sTexture_Editor );
			}
			set
			{
				m_sTexture_Editor = value;
			}
		}
		public string Texture_Space
		{
			get
			{
				return( m_sTexture_Space );
			}
			set
			{
				m_sTexture_Space = value;
			}
		}
		#endregion
	}
	public class GenericModule : Module
	{
		#region Properties
		private const string m_cTHRUST_SOUND = "Urquan_Thruster";
		private int m_nModuleID = -1;
		private int[,] m_naModuleTypes = null;
		private string m_sName = "";

		private bool[] m_baOn = new bool[ (int)enumMovementType.LastItem ];

		private double m_nEnergyStored = 0;
		private double m_nEnergyStoredThisFrame = 0;
		private double m_nEnergyStoreMax = 0;
		private double m_nEnergyStoreAbsorb = 0;
		private double m_nEnergyStoreRelease = 0;

		private double m_nEnergyGen = 0;

		private double m_nTurningThrust = 0;

		private double m_nConsumeEnergyAmount = 0;
		private bool m_nEnergyDebtPaid = false;

		private int m_nHUDLevel = 0;

		private double m_nCurLinearThrust = 0;
		private double m_nLinearThrust = 0;
		private double m_nLinearDragOnBreak = 0;
		private long m_nLinearThrust_PSystem = 0;
		private int m_nThruster_Sound_Index = -1;
		private string m_sThrustPMGUID = string.Empty;
		#endregion


		public GenericModule()
		{
			//Do not put code here
			throw new System.Exception( "Initializing chassis with wrong constructor." );
		}
		public GenericModule( ComplexEntity oParentEntity)
		{
			base.ParentEntity = oParentEntity;
		}
		public GenericModule( ComplexEntity oParentEntity,int nModuleID )
		{
			m_nModuleID = nModuleID;
			base.ParentEntity = oParentEntity;
		}
    	public override void CreatedAsNewItem()
		{
			string sDifficultyKey = "";
			string sModuleTypes = "";
			string[] saModules = null;
			string[] saFields = null;
			double nMin = 0, nMax = 0;
			int nRow = 0;
			int nDiffPart = 0;
			int nX = 0, nY = 0, nType = 0;
			DSGobTable oModuleTable = null;


			base.CreatedAsNewItem();

			if( m_nModuleID != -1 )
			{
				oModuleTable = DSResourceManager.GetGlobalInstance().GetGobTable( GobConstants.m_cTABLE_MODULE );
				//Find the row for this ID
				nRow = oModuleTable.FindRow( "GenericModuleID",m_nModuleID );
		
					 if( this.ItemLevel == enumItemLevel.Normal )	{ sDifficultyKey = "_Nrm";nDiffPart = 0; }
				else if( this.ItemLevel == enumItemLevel.Advanced )	{ sDifficultyKey = "_Adv";nDiffPart = 1; }
				else if( this.ItemLevel == enumItemLevel.Expert )	{ sDifficultyKey = "_Exp";nDiffPart = 2; }
 
				//Get the rendering components
				this.Texture_Editor	= DSMisc.TypeUtils.GetSafeStr( oModuleTable.GetData( "Texture",nRow ) );
				if( DSMisc.TypeUtils.GetSafeBool( oModuleTable.GetData( "ShowTextureInSpace",nRow ) ) == true )
				{
					this.Texture_Space = this.Texture_Editor;
				}
				base.MGridWidth		= DSMisc.TypeUtils.GetSafeInt( oModuleTable.GetData( "Width",nRow ) );
				base.MGridHeight	= DSMisc.TypeUtils.GetSafeInt( oModuleTable.GetData( "Height",nRow ) );

				//Get the Difficulty dependents
				m_sName	= DSMisc.TypeUtils.GetSafeStr( oModuleTable.GetData( "Name" + sDifficultyKey,nRow ) );
				this.MinLevel = DSMisc.TypeUtils.GetSafeLng( oModuleTable.GetData( "MinLevel" + sDifficultyKey,nRow ) );
				if( DSMisc.TypeUtils.GetSafeBool( oModuleTable.GetData( "StoreEnergy" + sDifficultyKey,nRow ) ) == true )
				{				
					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "StoreEnergy_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "StoreEnergy_Max" + sDifficultyKey,nRow ) );
					m_nEnergyStoreMax = DSMisc.GetRnd( nMin,nMax );
                    m_nEnergyStored = m_nEnergyStoreMax;

					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "StoreEnergyAbsorb_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "StoreEnergyAbsorb_Max" + sDifficultyKey,nRow ) );
					m_nEnergyStoreAbsorb = DSMisc.GetRnd( nMin,nMax );

					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "StoreEnergyRelease_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "StoreEnergyRelease_Max" + sDifficultyKey,nRow ) );
					m_nEnergyStoreRelease = DSMisc.GetRnd( nMin,nMax );
				}
				if( DSMisc.TypeUtils.GetSafeBool( oModuleTable.GetData( "GenerateEnergy" + sDifficultyKey,nRow ) ) == true )
				{				
					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "GenerateEnergy_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "GenerateEnergy_Max" + sDifficultyKey,nRow ) );
					m_nEnergyGen = DSMisc.GetRnd( nMin,nMax );
				}
				if( DSMisc.TypeUtils.GetSafeBool( oModuleTable.GetData( "TurningThrust" + sDifficultyKey,nRow ) ) == true )
				{				
					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "TurningThrust_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "TurningThrust_Max" + sDifficultyKey,nRow ) );
					m_nTurningThrust = DSMisc.GetRnd( nMin,nMax );
				}
				if( DSMisc.TypeUtils.GetSafeBool( oModuleTable.GetData( "LinearThruster" + sDifficultyKey,nRow ) ) == true )
				{				
					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "LinearThrust_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "LinearThrust_Max" + sDifficultyKey,nRow ) );
					m_nLinearThrust = DSMisc.GetRnd( nMin,nMax );
					
					nMin = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "LinearBreakThrust_Min" + sDifficultyKey,nRow ) );
					nMax = DSMisc.TypeUtils.GetSafeDbl( oModuleTable.GetData( "LinearBreakThrust_Max" + sDifficultyKey,nRow ) );
					m_nLinearDragOnBreak = DSMisc.GetRnd( nMin,nMax );

					m_nLinearThrust_PSystem = DSMisc.TypeUtils.GetSafeLng( oModuleTable.GetData( "LinearThrust_PSystem" + sDifficultyKey,nRow ) );
				}
				m_nHUDLevel = DSMisc.TypeUtils.GetSafeInt( oModuleTable.GetData( "HUDLevel" + sDifficultyKey,nRow ) );
				if( GetBoolPartFromModule( oModuleTable,"ConsumeEnergy",nRow,nDiffPart ) == true )
				{
					nMin = GetDblPartFromModule( oModuleTable,"ConsumeEnergy_Min",nRow,nDiffPart );
					nMax = GetDblPartFromModule( oModuleTable,"ConsumeEnergy_Max",nRow,nDiffPart );
					m_nConsumeEnergyAmount = DSMisc.GetRnd( nMin,nMax );
				}


				//Now load in our grid layouts
				sModuleTypes = DSMisc.TypeUtils.GetSafeStr( oModuleTable.GetData( "ModuleType",nRow ) );
				m_naModuleTypes = new int[ base.MGridWidth,base.MGridHeight ];
				saModules = DSMisc.Split( sModuleTypes ,";" );
				for( int i=0 ; i<saModules.Length-1 ; i++ )
				{
					try
					{
						saFields = DSMisc.Split( saModules[ i ],"," );
						nX		= DSMisc.TypeUtils.GetSafeInt( saFields[0] );
						nY		= DSMisc.TypeUtils.GetSafeInt( saFields[1] );
						nType	= DSMisc.TypeUtils.GetSafeInt( saFields[2] );
	
						if( nX >= 0 && nX < base.MGridWidth && nY >= 0 && nY < base.MGridHeight )
						{
							m_naModuleTypes[ nX,nY ] = nType;
						}
					}
					catch( System.Exception oEx )
					{
						DSMisc.ShowErrors( oEx );
					}
				}
			}
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			double nEnergyAvailable = 0;
			double nMaxAmountToTake = 0;
			double nTempEnergy = 0;


            base.Advance(oSession, nElapsedTime);

			if( this.ModuleIsAvailable() == true )
			{
				#region Energy Consumption
				nEnergyAvailable = this.ParentEntity.GetEnergyAvailable();
				if( nEnergyAvailable >= nElapsedTime * m_nConsumeEnergyAmount )
				{
					this.ParentEntity.TakeEnergy( nElapsedTime * m_nConsumeEnergyAmount );
					m_nEnergyDebtPaid = true;
				}
				else
				{
					m_nEnergyDebtPaid = false;
				}
				#endregion

				if( m_nEnergyDebtPaid == true )
				{
					#region Energy Storage
					if( m_nEnergyStored > GetMaxBatteryStore() )
					{
						m_nEnergyStored = GetMaxBatteryStore();
					}

					nMaxAmountToTake = Math.Max( GetMaxBatteryStore() - m_nEnergyStored,0 );
					nMaxAmountToTake = Math.Min( nMaxAmountToTake,GetMaxBatteryAbsorb() * nElapsedTime );

					//When taking energy, we don't want to take any from ourselves
					if( nMaxAmountToTake > 0 )
					{
						nTempEnergy = m_nEnergyStored;
						m_nEnergyStored = 0;
						m_nEnergyStoredThisFrame = base.ParentEntity.TakeEnergy( nMaxAmountToTake );
						m_nEnergyStored = m_nEnergyStoredThisFrame + nTempEnergy;
					}
					#endregion

					#region Thrust
					if( m_baOn[ (int)enumMovementType.Forward ] == true )
					{
						m_nCurLinearThrust = DSMisc.Min( m_nCurLinearThrust+nElapsedTime*100,GetForwardThrust() );
						//m_nCurLinearThrust = DSMisc.Max( m_nCurLinearThrust,GetMinThrust() );
					}
					else
					{
						m_nCurLinearThrust = 0;
					}
					#endregion
				}
			}

			////////////////////////////////////////////////////////
			//Thrust
			AdvanceThrustPM( nElapsedTime );
			AdvanceThrustSound();
		}
		private void AdvanceThrustSound()
		{
			if( this.ModuleIsAvailable() == true && Globals.Inst().IAmTheServer == false && m_nEnergyDebtPaid == true )
			{
                if (m_baOn[(int)enumMovementType.Forward] == true && m_nCurLinearThrust > 0)
				{
					//Sound isn't playing, start it
					if( m_nThruster_Sound_Index == -1 )
					{
                        m_nThruster_Sound_Index = Globals.PlaySound(m_cTHRUST_SOUND, this.ParentEntity.Location);
					}
					//Sound was playing but it stoped, restart it
					else if( Globals.Inst().GameEngine.DirectSound.SoundIsStillPlaying( m_cTHRUST_SOUND,m_nThruster_Sound_Index ) == false )
					{
                        m_nThruster_Sound_Index = Globals.PlaySound(m_cTHRUST_SOUND, this.ParentEntity.Location);
					}
				}
				else if( m_nThruster_Sound_Index != -1 )
				{
					Globals.Inst().GameEngine.DirectSound.StopPlayingSound( m_cTHRUST_SOUND,m_nThruster_Sound_Index );
                    m_nThruster_Sound_Index = -1;
				}
			}
			else if( m_nThruster_Sound_Index != -1 )
			{
				Globals.Inst().GameEngine.DirectSound.StopPlayingSound( m_cTHRUST_SOUND,m_nThruster_Sound_Index );
			}
		}
		private void AdvanceThrustPM( double nElapsedTime )
		{
			Vector2 vWorldPtUpperLeftModuleCorner = Vector2.Empty;
			Vector2 vWorldPos = Vector2.Empty;
			Vector2 vRotatedWorldPos = Vector2.Empty;
			DSParticleManager oPMan = null;
			double nAngle = 0;


			if( this.ModuleIsAvailable() == true && Globals.Inst().IAmTheServer == false && m_nEnergyDebtPaid == true )
			{
				if( m_baOn[ (int)enumMovementType.Forward ] == true )
				{
					if( m_sThrustPMGUID.Length == 0 )
					{
						if( m_nLinearThrust_PSystem == 1 )
						{
							oPMan = SLEntityEffect.StartShipThrusterThrust( new Vector2( 0,0 ),new Vector2( 0,0 ),this.ParentEntity.Location.ZoneID.ToString() );
							m_sThrustPMGUID = oPMan.GUID;
						}
						else if( m_nLinearThrust_PSystem == 2 )
						{
                            oPMan = SLEntityEffect.StartClassicShipThrusterThrust(new Vector2(0, 0), new Vector2(0, 0), this.ParentEntity.Location.ZoneID.ToString());
							m_sThrustPMGUID = oPMan.GUID;
						}
					}
					else
					{
						oPMan = Globals.Inst().GameEngine.ParticleSystem.GetParticleManager( m_sThrustPMGUID );
					}
					
					//Set our emmiter position
					if( oPMan != null )
					{
						oPMan.RunDown = false;

						vWorldPtUpperLeftModuleCorner = base.GetUpperLeftCornerWorldPt();

						vWorldPos = new Vector2( (float)( vWorldPtUpperLeftModuleCorner.X + ( base.MGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
							                     (float)( vWorldPtUpperLeftModuleCorner.Y + base.MGridHeight * GraphicConstants.m_cGRID_HEIGHT ) );
						vRotatedWorldPos = base.ParentEntity.GetRotatedWorldPt( vWorldPos );

						nAngle = base.ParentEntity.Angle + Math.PI / 2.0;
						oPMan.EmitterPosition = new Vector2( vRotatedWorldPos.X,vRotatedWorldPos.Y );
						oPMan.EmitterDirection = new Vector2( (float)Math.Cos( nAngle ),(float)Math.Sin( nAngle ) );

                        oPMan.EmitVelocity = 100 - 200.0f * ( base.ParentEntity.Vel.Length() / 500.0f );
                        oPMan.EmittersVelocity = Vector2.Empty;
						/*oPMan.EmittersVelocity = new Vector2( 
							base.ParentEntity.Vel.X * .65f + oPMan.EmitterDirection.X * 5,
							base.ParentEntity.Vel.Y * .65f + oPMan.EmitterDirection.Y * 5 );*/

						oPMan.ParticleLifeTimeInSeconds = 0.5 + 3.0 * ( m_nCurLinearThrust / 500.0 );
					}
					else 
					{
						TurnOffThrustPM();
					}
				}
				else if( m_sThrustPMGUID.Length > 0 )
				{
					TurnOffThrustPM();
				}
			}
			else if( m_sThrustPMGUID.Length > 0 )
			{
				TurnOffThrustPM();
			}
		}
		private void TurnOffThrustPM()
		{
			DSParticleManager oPMan = null;


			if( m_sThrustPMGUID.Length > 0 && 
				Globals.Inst().GameEngine != null )
			{
				oPMan = Globals.Inst().GameEngine.ParticleSystem.GetParticleManager( m_sThrustPMGUID );
				if( oPMan != null )
				{
					oPMan.RunDown = true;
				}
				m_sThrustPMGUID = "";
			}
		}

		public override ArrayList GetHoverRenderText()
		{
			string sName = string.Empty;
			ArrayList oProperties = base.GetHoverRenderText();


			if( m_nEnergyStoreMax > 0 )
			{
				oProperties.Add( "Max Energy Capacity of " + this.GetMaxBatteryStore().ToString( "0" ) );
				oProperties.Add( "Energy Store Rate of " + this.GetMaxBatteryAbsorb().ToString( "0.0" ) + " / s" );
				oProperties.Add( "Energy Draw Rate of " + this.GetMaxBatteryAbsorb().ToString( "0.0" ) + " / s" );
			}

			if( m_nEnergyGen > 0 )
			{
				oProperties.Add( GetEnergyProduced().ToString( "0" ) + " Energy Produced / s" );
			}

			if( m_nTurningThrust > 0 )
			{
				oProperties.Add( GetAngularThrust().ToString( "0" ) + " Angular thrust / s" );
			}

			if( m_nLinearThrust > 0 )
			{
				oProperties.Add( GetForwardThrust().ToString( "0" ) + " Linear thrust / s" );
				oProperties.Add( GetBackwardThrust().ToString( "0" ) + " Breaking thrust / s" );
			}

			if( m_nHUDLevel > 0 )
			{
				if( (enumHUDDetail)m_nHUDLevel == enumHUDDetail.None )
				{
					sName = "None";
				}
				else if( (enumHUDDetail)m_nHUDLevel == enumHUDDetail.Name )
				{
					sName = "Name";
				}
				else if( (enumHUDDetail)m_nHUDLevel == enumHUDDetail.HP3ColorIndicator )
				{
					sName = "3-Color HP";
				}
				else if( (enumHUDDetail)m_nHUDLevel == enumHUDDetail.HPBar )
				{
					sName = "Hit Point Bar";
				}
				else if( (enumHUDDetail)m_nHUDLevel == enumHUDDetail.HPShieldBar )
				{
					sName = "HP/Shield Bar";
				}
				else if( (enumHUDDetail)m_nHUDLevel == enumHUDDetail.HPShieldEnergy )
				{
					sName = "HP/Shield/Energy Bar";
				}
				oProperties.Add( "H.U.D. Level '" + sName + "'" );
			}

			if( m_nConsumeEnergyAmount > 0 )
			{
				oProperties.Add( "Costs " + m_nConsumeEnergyAmount.ToString( "0.0" ) + " per s" );
			}

			return( oProperties );
		}

		private double GetForwardThrust()
		{
			return( m_nLinearThrust * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		private double GetBackwardThrust()
		{
			return( m_nLinearDragOnBreak * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		private double GetMaxBatteryStore()
		{
			return( m_nEnergyStoreMax * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		private double GetMaxBatteryAbsorb()
		{
			return( m_nEnergyStoreAbsorb * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		private double GetMaxBatteryDrawPerSecond()
		{
			return( m_nEnergyStoreRelease * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
    	private double GetAngularThrust()
		{
			return( m_nTurningThrust * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		private double GetEnergyProduced()
		{
			return( m_nEnergyGen * ( base.StructurePoints / base.MaxStructurePoints ) );
		}

		#region New Gob access functions
		private bool GetBoolPartFromModule( DSGobTable oModuleTable,string sColumnKey,int nRow,int nPart )
		{
			bool bValue = false;
			string sValue = string.Empty;
			string[] saParts = null;


			sValue = DSMisc.TypeUtils.GetSafeStr( oModuleTable.GetData( sColumnKey,nRow ) );
			saParts = DSMisc.Split( sValue,";" );

			if( saParts.Length > nPart )
			{
				bValue = DSMisc.TypeUtils.GetSafeBool( saParts[ nPart ] );
			}
			

			return( bValue );
		}
		private double GetDblPartFromModule( DSGobTable oModuleTable,string sColumnKey,int nRow,int nPart )
		{
			double nValue = 0.0;
			string sValue = string.Empty;
			string[] saParts = null;


			sValue = DSMisc.TypeUtils.GetSafeStr( oModuleTable.GetData( sColumnKey,nRow ) );
			saParts = DSMisc.Split( sValue,";" );

			if( saParts.Length > nPart )
			{
				nValue = DSMisc.TypeUtils.GetSafeDbl( saParts[ nPart ] );
			}
			

			return( nValue );
		}
		#endregion

		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			enumModuleType nRetVal = enumModuleType.OpenSpace;

			if( nXOffset >= 0 && nXOffset < base.MGridWidth && nYOffset <= 0 && nYOffset > -base.MGridHeight )
			{
				nRetVal = (enumModuleType)m_naModuleTypes[ nXOffset,-nYOffset ];
			}

			return( nRetVal );
		}
		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			double nSign = 0;
			double nValue = 0;
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );

			if( this.ModuleIsAvailable() == true && m_nEnergyDebtPaid == true )
			{
				if( nPropertyToGet == enumEntProperties.Movement_AngularThrust )
				{
						 if( m_baOn[ (int)enumMovementType.TurnLeft ] == true ){ nSign = -1; }
					else if( m_baOn[ (int)enumMovementType.TurnRight ] == true ){ nSign = 1; }

					if( m_baOn[ (int)enumMovementType.TurnLeft ] == true ||
						m_baOn[ (int)enumMovementType.TurnRight ] == true )
					{
						nValue = GetAngularThrust() * nSign;
					}				
				}
				/*else if( nPropertyToGet == enumEntProperties.Movement_EtherialAngularDrag &&
					m_baOn[ (int)enumMovementType.Backward ] == true )
				{
					nValue = GetAngularThrust();
				}*/
				else if( nPropertyToGet == enumEntProperties.Energy_Available ||
					nPropertyToGet == enumEntProperties.Energy_CurrentStorage)
				{
					nValue = m_nEnergyStored;
				}
				else if( nPropertyToGet == enumEntProperties.Energy_MaxStorage )
				{
					nValue = GetMaxBatteryStore();
				}
				else if( nPropertyToGet == enumEntProperties.Movement_EtherialLinearDrag &&
						 m_baOn[ (int)enumMovementType.Backward ] == true )
				{
					nValue = GetBackwardThrust();
				}
				else if( nPropertyToGet == enumEntProperties.Movement_LinearThrust &&
						 m_baOn[ (int)enumMovementType.Forward ] == true )
				{
					nValue = m_nCurLinearThrust;
				}
				else if( nPropertyToGet == enumEntProperties.Movement_LinearThrust_Max )
				{
					nValue = GetForwardThrust();
				}
				else if( nPropertyToGet == enumEntProperties.Energy_Produced )
				{
					nValue = GetEnergyProduced();
				}
				else if( nPropertyToGet == enumEntProperties.HUD_HPDetail )
				{
					nValue = DSMisc.Max( m_nHUDLevel,nBaseValue );
					nBaseValue = 0;
				}
			}

			return( nBaseValue + nValue );
		}
		public override double TakeEnergy(double nAmountToTake)
		{
			double nAmtToTake = 0;

			nAmtToTake = Math.Min( m_nEnergyStored,nAmountToTake );
			nAmtToTake = Math.Min( nAmtToTake,this.ParentEntity.GetEnergyAvailable() );
			//The amount taken this frame allows us to draw any amount the battery has sucked up that would have been used.  
			m_nEnergyStored -= nAmtToTake;
			m_nEnergyStoredThisFrame = Math.Max( 0,m_nEnergyStoredThisFrame-nAmtToTake );

			return( nAmtToTake );
		}
        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.LeftArrow )
			{
				m_baOn[ (int)enumMovementType.TurnLeft ] = bPressed;
			}
			else if( oKey == Key.RightArrow )
			{
				m_baOn[ (int)enumMovementType.TurnRight ] = bPressed;
			}
			else if( oKey == Key.UpArrow )
			{
				m_baOn[ (int)enumMovementType.Forward ] = bPressed;
			}
			else if( oKey == Key.DownArrow )
			{
				m_baOn[ (int)enumMovementType.Backward ] = bPressed;
			}
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,base.Texture_Editor );
			//oSerialize.Set( 2,base.Texture_Space );
			oSerialize.Set( 3,base.MGridWidth );
			oSerialize.Set( 4,base.MGridHeight );
			oSerialize.Set( 5,m_sName );
			oSerialize.SetList( 6 );

			for( int nX=0 ; nX<base.MGridWidth ; nX++ )
			{
				for( int nY=0 ; nY<base.MGridHeight ; nY++ )
				{
					if( m_naModuleTypes[ nX,nY ] != 0 )
					{
						oSerialize.SetListItem( 6,nX );
						oSerialize.SetListItem( 6,nY );
						oSerialize.SetListItem( 6,m_naModuleTypes[ nX,nY ] );
					}
				}
			}

			oSerialize.Set( 7,( base.Texture_Space.Length > 0 ) );
			oSerialize.Set( 8,m_nEnergyStored );
			oSerialize.Set( 9,m_nEnergyStoredThisFrame );
			oSerialize.Set( 10,m_nEnergyStoreMax );
			oSerialize.Set( 11,m_nEnergyStoreAbsorb );
			oSerialize.Set( 12,m_nEnergyStoreRelease );
			oSerialize.Set( 13,m_nEnergyGen );
			oSerialize.Set( 14,m_nTurningThrust );
			oSerialize.Set( 15,m_nLinearThrust );
			oSerialize.Set( 16,m_nLinearDragOnBreak );
			oSerialize.Set( 17,m_nLinearThrust_PSystem );
			oSerialize.Set( 18,m_nConsumeEnergyAmount );
			oSerialize.Set( 19,m_nHUDLevel );			
			 

			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			ArrayList oArrayList = null;

			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			base.Texture_Editor	= oSerialize.GetString( 1 );
			//base.Texture_Space	= oSerialize.GetString( 2 );
			base.MGridWidth		= oSerialize.GetInt( 3 );
			base.MGridHeight	= oSerialize.GetInt( 4 );
			m_sName				= oSerialize.GetString( 5 );

			oArrayList = oSerialize.GetList( 6 );
			m_naModuleTypes = new int[ base.MGridWidth,base.MGridHeight ];
			for( int i=0 ; i<oArrayList.Count ; i+=3 )
			{
				m_naModuleTypes[ Convert.ToInt32( oArrayList[ i ] ),
								 Convert.ToInt32( oArrayList[ i+1 ] ) ] = Convert.ToInt32( oArrayList[ i+2 ] );
			}

			if( oSerialize.GetBool( 7 ) == true )
			{
				base.Texture_Space = base.Texture_Editor;
			}

			m_nEnergyStored			= oSerialize.GetDouble( 8 );
			m_nEnergyStoredThisFrame= oSerialize.GetDouble( 9 );
			m_nEnergyStoreMax		= oSerialize.GetDouble( 10 );
			m_nEnergyStoreAbsorb	= oSerialize.GetDouble( 11 );
			m_nEnergyStoreRelease	= oSerialize.GetDouble( 12 );
			m_nEnergyGen			= oSerialize.GetDouble( 13 );
			m_nTurningThrust		= oSerialize.GetDouble( 14 );
			m_nLinearThrust			= oSerialize.GetDouble( 15 );
			m_nLinearDragOnBreak	= oSerialize.GetDouble( 16 );
			m_nLinearThrust_PSystem	= oSerialize.GetLong( 17 );
			m_nConsumeEnergyAmount	= oSerialize.GetDouble( 18 );
			m_nHUDLevel				= oSerialize.GetInt( 19 );
		}
		public override double DealDamage(double nAmount, Region oDamageWhere)
		{
			double nDamageLeftOver = base.DealDamage( nAmount, oDamageWhere );
			AdvanceThrustPM( 0 );
			return( nDamageLeftOver );
		}
		public override void UnAssign()
		{
			base.UnAssign();
			AdvanceThrustPM( 0 );
		}

        
		#region Properties
		public override string Name
		{
			get
			{
				return( m_sName );
			}
		}
		#endregion
	}

	public class XMissileLauncher : Module
	{
		#region Properties
		private bool m_bFiring = false;
		private double m_nTimeSinceLastShot = 30;

		private int m_nMinDamage = 350;
		private int m_nMaxDamage = 350;
		private double m_nRechargeTime = 4;
		private double m_nECostPerShot = 10;
		#endregion


		public XMissileLauncher()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public XMissileLauncher( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 1;
			base.MGridHeight = 3;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "XMissile_Launcher";
			this.Texture_Space = "XMissile_Launcher";
		}
		public override void CreatedAsNewItem()
		{
			base.CreatedAsNewItem();

			if( this.ItemLevel == enumItemLevel.Normal )
			{
				m_nMinDamage = DSMisc.GetRnd( 200,250 );
				m_nMaxDamage = m_nMinDamage + DSMisc.GetRnd( 50,150 );
				m_nRechargeTime = DSMisc.GetRnd() * 2.0 - 1.0 + 4.0;
				m_nECostPerShot = DSMisc.GetRnd() * 10.0 - 5.0 + 10.0;
			}
			else if( this.ItemLevel == enumItemLevel.Advanced )
			{
				m_nMinDamage = DSMisc.GetRnd( 450,550 );
				m_nMaxDamage = m_nMinDamage + DSMisc.GetRnd( 100,250 );
				m_nRechargeTime = DSMisc.GetRnd() * 2.0 - 1.0 + 4.0;
				m_nECostPerShot = DSMisc.GetRnd() * 20.0 - 10.0 + 55.0;
			}
			else if( this.ItemLevel == enumItemLevel.Expert )
			{
				m_nMinDamage = DSMisc.GetRnd( 1050,1550 );
				m_nMaxDamage = m_nMinDamage + DSMisc.GetRnd( 200,550 );
				m_nRechargeTime = DSMisc.GetRnd() * 2.0 - 1.0 + 4.0;
				m_nECostPerShot = DSMisc.GetRnd() * 80.0 - 40.0 + 150.0;
			}			
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);

			if( base.StructurePoints > 0 )
			{
				m_nTimeSinceLastShot += nElapsedTime;

				if( m_bFiring == true )
				{				
					Fire(oSession);
				}
			}
		}
		public void Fire(Session oSession)
		{
			double nLaunchVelocity = 60;
			Vector2 vModuleWorldPtUpperLeftCorner = Vector2.Empty;
			Vector2 vLaunchWorldPt = Vector2.Empty;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;
            Zone oMyZone = null;
            XMissile oShot = null;


            if (this.ModuleIsAvailable() == true && this.CanLaunchProjectile() == true && m_nTimeSinceLastShot > TimeBetweenShots())
			{
				if( base.ParentEntity.GetEnergyAvailable() > m_nECostPerShot )
				{
					base.ParentEntity.TakeEnergy( m_nECostPerShot );

					//Add our missile
                    oShot = new XMissile();
					vModuleWorldPtUpperLeftCorner = GetUpperLeftCornerWorldPt();
					vLaunchWorldPt = new Vector2( 
						(float)( vModuleWorldPtUpperLeftCorner.X + ( base.MGridWidth * .5 ) * GraphicConstants.m_cGRID_WIDTH ),
						(float)( vModuleWorldPtUpperLeftCorner.Y + ( base.MGridHeight * .10 ) * GraphicConstants.m_cGRID_HEIGHT ) );
					vRotatedLaunchWorldPt = base.ParentEntity.GetRotatedWorldPt( vLaunchWorldPt );

                    oShot.Angle = base.ParentEntity.Angle;// -Math.PI / 2.0;
					oShot.Vel = new Vector2( (float)( Math.Cos( oShot.Angle-Math.PI / 2.0 ) * nLaunchVelocity + base.ParentEntity.Vel.X ),
						                     (float)( Math.Sin( oShot.Angle-Math.PI / 2.0 ) * nLaunchVelocity + base.ParentEntity.Vel.Y ) );
					oShot.Pos = vRotatedLaunchWorldPt;
                    oShot.Location.ZoneID = this.ParentEntity.Location.ZoneID;
					oShot.OwnerSocketID = this.ParentEntity.OwnerSocketID;
					oShot.ImpactDamage = DSMisc.GetRnd( m_nMinDamage,m_nMaxDamage );
					oShot.ExplodeDamage = 0;


					//Add him to the session
                    m_nTimeSinceLastShot = 0;
                    oMyZone = oSession.Zones[base.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];
                    oMyZone.AddEntity(oSession,oShot, true);
				}
				else if( Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1 )
				{
                    Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
				}
			}
		}
		private double TimeBetweenShots()
		{
			return( 4 + 8 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) ) ); 
		}

		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();

			oProperties.Add( "Damage: " + m_nMinDamage.ToString( "0" ) + " - " + m_nMaxDamage.ToString( "0" ) );
            if (TimeBetweenShots() != m_nRechargeTime)
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " (" + m_nRechargeTime.ToString("0.0") + ") seconds");
            }
            else
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " seconds");
            }	
			oProperties.Add( "Costs " + m_nECostPerShot.ToString( "0.0" ) + " per shot" );

			return( oProperties );
		}

		public override void KeyProcess(Session oSession,Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.M &&
				Globals.Inst().OurShip != null && 
				Globals.Inst().OurShip.PKey == this.ParentEntity.PKey )
			{
				if( bPressed == true )
				{
					Fire(oSession);
				}
				m_bFiring = bPressed;
			}
		}
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 && nYOffset == 0 ) ||
				( nXOffset == 0 && nYOffset == 0-1 ) ||
				( nXOffset == 0 && nYOffset == 0-2 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( nXOffset == 0 && nYOffset == 0 )
			{
				return( enumModuleType.OpenSpace );
			}
			else if( nXOffset == 0 && ( nYOffset == -1 || nYOffset == -2 ) ) 
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}

		public override DSSerialize Serialize(Session oSession)
		{
			//ItemSpecialTrait oLoopItem = null;
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_bFiring );
			oSerialize.Set( 2,m_nMinDamage );
			oSerialize.Set( 3,m_nMaxDamage );
			oSerialize.Set( 4,m_nRechargeTime );
			oSerialize.Set( 5,m_nECostPerShot );
			

			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_bFiring = oSerialize.GetBool( 1 );

			if( oSerialize.Version >= 2 )
			{
				m_nMinDamage = oSerialize.GetInt( 2 );
				m_nMaxDamage = oSerialize.GetInt( 3 );
				m_nRechargeTime = oSerialize.GetDouble( 4 );
				m_nECostPerShot = oSerialize.GetDouble( 5 );
			}
		}


		#region properties
		public bool Firing
		{
			get
			{
				return( m_bFiring );
			}
			set
			{
				m_bFiring = value;
			}
		}
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		public override string Name
		{
			get
			{
				string sName = "";

				if( this.ItemLevel == enumItemLevel.Normal )
				{
					sName = "K4 Rocket";
				}
				else if( this.ItemLevel == enumItemLevel.Advanced )
				{
					sName = "Neutreno Missile";
				}
				else if( this.ItemLevel == enumItemLevel.Expert )
				{
					sName = "Fission Torpedo";
				}			

				return( sName );
			}
		}
		#endregion
	}
	public class UrquanPlasmaLauncher : Module
	{
		#region Properties
		private bool m_bFiring = false;
		private double m_nTimeSinceLastShot = 30;

		private int m_nMinDamage = 40;
		private int m_nMaxDamage = 40;
		private double m_nRechargeTime = .5;
		private double m_nECostPerShot = 20;
		#endregion


		public UrquanPlasmaLauncher()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public UrquanPlasmaLauncher( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 3;
			base.MGridHeight = 4;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "Plasma_Launcher";
			this.Texture_Space = "Plasma_Launcher";
		}
		public override void CreatedAsNewItem()
		{
			base.CreatedAsNewItem();

			if( this.ItemLevel == enumItemLevel.Normal )
			{
				m_nMinDamage = DSMisc.GetRnd( 20,40 );
				m_nMaxDamage = m_nMinDamage + DSMisc.GetRnd( 10,20 );
				m_nRechargeTime = DSMisc.GetRnd() * .4 - .2 + .5;
				m_nECostPerShot = DSMisc.GetRnd() * 10.0 - 5.0 + 6.0;
			}
			else if( this.ItemLevel == enumItemLevel.Advanced )
			{
				m_nMinDamage = DSMisc.GetRnd( 90,110 );
				m_nMaxDamage = m_nMinDamage + DSMisc.GetRnd( 30,60 );
				m_nRechargeTime = DSMisc.GetRnd() * .4 - .2 + .3;
				m_nECostPerShot = DSMisc.GetRnd() * 20.0 - 10.0 + 30.0;
			}
			else if( this.ItemLevel == enumItemLevel.Expert )
			{
				m_nMinDamage = DSMisc.GetRnd( 150,190 );
				m_nMaxDamage = m_nMinDamage + DSMisc.GetRnd( 50,80 );
				m_nRechargeTime = DSMisc.GetRnd() * .4 - .2 + .1;
				m_nECostPerShot = DSMisc.GetRnd() * 50.0 - 25.0 + 120.0;
			}			
		}

		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();

			oProperties.Add( "Damage: " + m_nMinDamage.ToString( "0" ) + " - " + m_nMaxDamage.ToString( "0" ) );
            if (TimeBetweenShots() != m_nRechargeTime)
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " (" + m_nRechargeTime.ToString("0.0") + ") seconds");
            }
            else
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " seconds");
            }			
			oProperties.Add( "Costs " + m_nECostPerShot.ToString( "0.0" ) + " per shot" );

			return( oProperties );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);
			
			if( base.StructurePoints > 0 )
			{
				m_nTimeSinceLastShot += nElapsedTime;

				if( m_bFiring == true )
				{				
					Fire(oSession);
				}
			}
		}
		public void Fire(Session oSession)
		{
			double nLaunchVelocity = 300;
			Vector2 vModuleWorldPtUpperLeftCorner = Vector2.Empty;
			Vector2 vLaunchWorldPt = Vector2.Empty;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;
            Zone oMyZone = null;
			UrquanPlasmaProjectile oShot = new UrquanPlasmaProjectile();


            if (this.ModuleIsAvailable() == true && this.CanLaunchProjectile() == true && m_nTimeSinceLastShot > TimeBetweenShots())
			{
				if( base.ParentEntity.GetEnergyAvailable() > m_nECostPerShot )
				{
					base.ParentEntity.TakeEnergy( m_nECostPerShot );
					 
					//Add our missile
					vModuleWorldPtUpperLeftCorner = GetUpperLeftCornerWorldPt();
					vLaunchWorldPt = new Vector2( 
						(float)( vModuleWorldPtUpperLeftCorner.X + ( base.MGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
						(float)( vModuleWorldPtUpperLeftCorner.Y /*+ ( base.MGridHeight * .25 ) * GraphicConstants.m_cGRID_HEIGHT*/ ) );
					vRotatedLaunchWorldPt = base.ParentEntity.GetRotatedWorldPt( vLaunchWorldPt );

					oShot.Angle = base.ParentEntity.Angle - Math.PI / 2.0;
					oShot.Vel = new Vector2( (float)( Math.Cos( oShot.Angle ) * nLaunchVelocity + base.ParentEntity.Vel.X ),
						(float)( Math.Sin( oShot.Angle ) * nLaunchVelocity + base.ParentEntity.Vel.Y ) );
					oShot.Pos = vRotatedLaunchWorldPt;				
					oShot.OwnerSocketID = this.ParentEntity.OwnerSocketID;
					oShot.ImpactDamage = DSMisc.GetRnd( m_nMinDamage,m_nMaxDamage );
					oShot.ExplodeDamage = 0;
                    oShot.Location.ZoneID = this.ParentEntity.Location.ZoneID;

					//Add our graphical effects for the luanc and missile flare
					/*oPMan = Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem( 
								System.Drawing.Color.Red,System.Drawing.Color.Black,15,
								10,new Vector3( oShot.Pos.X,oShot.Pos.Y,0),new Vector3( oShot.Vel.X,oShot.Vel.Y,0 ) );
					oPMan.MaxNumParticlesToEmit = 50;
					oPMan = Globals.Inst().GameEngine.ParticleSystem.CreateParticleSystem( 
								System.Drawing.Color.Red,System.Drawing.Color.Black,15,
								10,new Vector3( oShot.Pos.X,oShot.Pos.Y,0),new Vector3( -oShot.Vel.X,-oShot.Vel.Y,0 ) );
					oShot.ThrustPMGUID = oPMan.PKey;*/

					//Add him to the session
					m_nTimeSinceLastShot = 0;
                    oMyZone = oSession.Zones[base.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];
                    oMyZone.AddEntity(oSession,oShot, true);
				}
				else if( Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1 )
				{
                    Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
				}
			}
		}
		private double TimeBetweenShots()
		{
			double nBaseRate = 0;
			double nMod = 0;

			if( this.Chassis != null )
			{
				nBaseRate = this.Chassis.GetPropertyTotal( enumEntProperties.Weapon_SecondsToRecharge,false );
				nMod = this.Chassis.GetPropertyTotal( enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt,false );
			}
			nBaseRate += this.GetPropertyTotal( enumEntProperties.Weapon_SecondsToRecharge,false );
			nMod += 1 + this.GetPropertyTotal( enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt,false );
			nBaseRate = nBaseRate * nMod;
 
			return( nBaseRate ); 
		}

        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.Space &&
				Globals.Inst().OurShip != null && 
				Globals.Inst().OurShip.PKey == this.ParentEntity.PKey )
			{
				if( bPressed == true )
				{
					Fire(oSession);
				}
				m_bFiring = bPressed;
			}
		}
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( nXOffset >= 0 && nXOffset <= 2 &&
				nYOffset <= 0 && nYOffset >= -3 )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( /*nXOffset == 0 && */nYOffset == 0 )
			{
				return( enumModuleType.OpenSpace );
			}
			else if( nXOffset >= 0 && nXOffset <= 2 &&
				nYOffset <= -1 && nYOffset >=-3 ) 
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}
		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );
			double nMyValue = 0;

			if( nPropertyToGet == enumEntProperties.Weapon_SecondsToRecharge )
			{
				nMyValue = m_nRechargeTime + 15 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) );
			}

			return( nBaseValue + nMyValue );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_bFiring );
			oSerialize.Set( 2,m_nMinDamage );
			oSerialize.Set( 3,m_nMaxDamage );
			oSerialize.Set( 4,m_nRechargeTime );
			oSerialize.Set( 5,m_nECostPerShot );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_bFiring = false;//oSerialize.GetBool( 1 );
			m_nMinDamage = oSerialize.GetInt( 2 );
			m_nMaxDamage = oSerialize.GetInt( 3 );
			m_nRechargeTime = oSerialize.GetDouble( 4 );
			m_nECostPerShot = oSerialize.GetDouble( 5 );
		}


		#region properties
		public bool Firing
		{
			get
			{
				return( m_bFiring );
			}
			set
			{
				m_bFiring = value;
			}
		}
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		public override string Name
		{
			get
			{
				string sName = "";

				if( this.ItemLevel == enumItemLevel.Normal )
				{
					sName = "Plasma Launcher";
				}
				else if( this.ItemLevel == enumItemLevel.Advanced )
				{
					sName = "Plasma Cannon";
				}
				else if( this.ItemLevel == enumItemLevel.Expert )
				{
					sName = "Molten Caster";
				}			

				return( sName );
			}
		}
		#endregion
	}
	public class Radar : Module
	{
		#region Properties
		private bool m_bEnergyBillPaid = false;
		private double m_nTimeSinceLastEnergyCheck = m_cMINTIMEBETWEENENERGYCHECK;

		private const double m_cMINTIMEBETWEENENERGYCHECK = 2;
		#endregion


		public Radar()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public Radar( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 1;
			base.MGridHeight = 1;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "Radar";
			this.Texture_Space = "";
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			double nEnergyFound = 0;


            base.Advance(oSession, nElapsedTime);

			m_nTimeSinceLastEnergyCheck += nElapsedTime;

			if( base.ParentEntity != null &&
				base.ParentEntity.GetEnergyAvailable() >= 5*nElapsedTime &&
				(
				m_bEnergyBillPaid == true ||
				m_nTimeSinceLastEnergyCheck >= m_cMINTIMEBETWEENENERGYCHECK 
				)
				)
			{
				m_nTimeSinceLastEnergyCheck = 0;
				m_bEnergyBillPaid = true;
				nEnergyFound = base.ParentEntity.TakeEnergy( 5*nElapsedTime );
			}
			else
			{
				m_bEnergyBillPaid = false;
			}
		}
		public override void Render( RenderSettings oRenderSettings,Chassis oParentChassis )
		{
			Vector2 vRelativePos = Vector2.Empty;
			Rectangle oTargetRect = Rectangle.Empty;
            Zone oMyZone = null;
			string sTexture = "";
			double nCenterX = 85, nCenterY = 92;


			if( oRenderSettings.RenderType == enumRenderType.HUD && m_bEnergyBillPaid == true &&
				this.ParentEntity.IsDead == false && this.StructurePoints > 0 && 
				this.ParentEntity == Globals.Inst().OurShip )
			{
				Globals.Inst().GameEngine.RenderTexture2D( 
					"HUD",
					System.Drawing.Rectangle.Empty,
					new Rectangle( 0,0,0,0 ),
					Vector2.Empty,//Rotate around
					0,0,false,System.Drawing.Color.White.ToArgb() );

				//Now render nearby guys
                oMyZone = Globals.Inst().Session.Zones[this.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];
				foreach( Entity oLoopEntity in oMyZone.Entitys.Values )
				{
					vRelativePos = oLoopEntity.Pos - oParentChassis.ParentEntity.Pos;
					if( oLoopEntity.IsVisible == true &&
						oLoopEntity.IsInPlay == true &&
						vRelativePos.X > -3000 && vRelativePos.X < 3000 &&
						vRelativePos.Y > -3000 && vRelativePos.Y < 3000 )
					{
						oTargetRect = new Rectangle( 
							(int)( nCenterX + vRelativePos.X / 100.0 ),
							(int)( nCenterY + vRelativePos.Y / 100.0 ),
							4,4 );

						if( oLoopEntity.GetType() == typeof( ComplexEntity ) )
						{
							sTexture = "Radar_Marker_Player";
						}
                        else if( oLoopEntity.GetType() == typeof( Portal ) )
                        {
                            sTexture = "Radar_Marker_Portal";
                            oTargetRect.Width = 6;
                            oTargetRect.Height = 6;
                        }
						else if( oLoopEntity.EntityType == enumEntityType.Ship )
						{
							sTexture = "Radar_Marker_Shagot";
						}
						else if( oLoopEntity.EntityType == enumEntityType.Astrological )//|| oLoopEntity.EntityType == enumEntityType.Debris )
						{
							sTexture = "Radar_Marker_Asteroid";
						}
						else
						{
							sTexture = "";//"Radar_Marker_Unknown";
						}

						if( sTexture.Length > 0 )
						{
							Globals.Inst().GameEngine.RenderTexture2D( 
								sTexture,System.Drawing.Rectangle.Empty,oTargetRect,
								Vector2.Empty,//Rotate around
								0,0,false,System.Drawing.Color.White.ToArgb() );
						}
					}
				}
			}
		}
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			return( nXOffset == 0 && nYOffset == 0 );
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( MGridIsInUse( nXOffset,nYOffset ) == true )
			{
				return( enumModuleType.HUD );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
		}


		#region Properties
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.HUD );
			}
		}
		#endregion
	}
	public class CloakingDevice : Module
	{
		#region Properties
		private enum enumCloakingStatus
		{
			Uncloaked,
			Cloaking,
			Cloaked,
			UnCloaking
		}

		private enumCloakingStatus m_nStatus = enumCloakingStatus.Uncloaked;
		private double m_nPercentCloaking = 0;
		private bool m_bHavePlayedCloakingSound = false;
		private bool m_bHavePlayedDeCloakingSound = false;
		#endregion


		public CloakingDevice()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}

		public CloakingDevice( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 2;
			base.MGridHeight = 2;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "CloakingDevice";
			this.Texture_Space = "";
		}


        public override void Advance(Session oSession, double nElapsedTime)
		{		
			double nEnergyNeeds = GetCloakingEnergyNeeds() * nElapsedTime;


            base.Advance(oSession, nElapsedTime);

			//Are we dead?  If so set to uncloaking
			if( m_nStatus != enumCloakingStatus.Uncloaked &&
				( this.ParentEntity.IsDead == true || this.StructurePoints <= 0 ) )
			{
				m_nStatus = enumCloakingStatus.UnCloaking;
			}
			else if( m_nStatus == enumCloakingStatus.Cloaking || m_nStatus == enumCloakingStatus.Cloaked )
			{
				if( base.ParentEntity.GetEnergyAvailable() > nEnergyNeeds )
				{
					base.ParentEntity.TakeEnergy( nEnergyNeeds );
				}
				else
				{
					if( Globals.Inst().IAmTheServer == false )
					{
                        Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
					}
					m_nStatus = enumCloakingStatus.UnCloaking;
				}
			}


			if( m_nStatus == enumCloakingStatus.Cloaking )
			{
				if( m_bHavePlayedCloakingSound == false && Globals.Inst().IAmTheServer == false )
				{
					m_bHavePlayedCloakingSound = true;
                    Globals.PlaySound("CloakingDevice_Cloak", this.ParentEntity.Location);
				}

				m_nPercentCloaking += GetCloakingSpeedChange() * nElapsedTime;
				if( m_nPercentCloaking >= 1 )
				{
					m_nPercentCloaking = 1;
					m_nStatus = enumCloakingStatus.Cloaked;
					m_bHavePlayedCloakingSound = false;
				}
			}
			else if( m_nStatus == enumCloakingStatus.UnCloaking )
			{
				if( m_bHavePlayedDeCloakingSound == false && Globals.Inst().IAmTheServer == false )
				{
					m_bHavePlayedDeCloakingSound = true;
                    Globals.PlaySound("CloakingDevice_DeCloak", this.ParentEntity.Location);
				}

				m_nPercentCloaking -= GetCloakingSpeedChange() * nElapsedTime;
				if( m_nPercentCloaking <= 0 )
				{
					m_nPercentCloaking = 0;
					m_nStatus = enumCloakingStatus.Uncloaked;
					m_bHavePlayedDeCloakingSound = false;
				}
			}

			if( m_nStatus == enumCloakingStatus.Cloaked )
			{
				this.ParentEntity.IsVisible = false;
			}
			else
			{
				this.ParentEntity.IsVisible = true;
			}
		}

		private double GetCloakingSpeedChange()
		{
			return( 1 * ( base.StructurePoints / base.MaxStructurePoints ) );
		}

		private double GetCloakingEnergyNeeds()
		{
			//Normal costs, plus mass modifier, plus damage penalty
			return( 50 + this.ParentEntity.GetPropertyTotal( enumEntProperties.Structure_Mass ) / 40.0 + 
					300 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) ) );
		}

        
		public override void PreRenderRenderSettingsChange(RenderSettings oRenderSettings)
		{
			base.PreRenderRenderSettingsChange( oRenderSettings );

			if( m_nStatus == enumCloakingStatus.Cloaked )
			{
				if( Globals.Inst().OurShip != null &&
					this.ParentEntity.PKey == Globals.Inst().OurShip.PKey )
				{
					oRenderSettings.BaseDrawColor = System.Drawing.Color.Red;
					oRenderSettings.PercentTransparent = .5;
				}
				else
				{
					oRenderSettings.PercentTransparent = 1;
				}
			}
			else if( m_nStatus != enumCloakingStatus.Uncloaked )
			{
				if( m_nPercentCloaking > 0 && m_nPercentCloaking < .3 )
				{
					oRenderSettings.BaseDrawColor = System.Drawing.Color.FromArgb( 
						(int)( ( ( .3 - m_nPercentCloaking ) / .3 ) * 255 ),
						(int)( ( ( .3 - m_nPercentCloaking ) / .3 ) * 255 ),
						255 );
				}
				else if( m_nPercentCloaking > .3 )
				{
					oRenderSettings.BaseDrawColor = System.Drawing.Color.FromArgb( 0,0,255 );
					oRenderSettings.PercentTransparent = ( m_nPercentCloaking - .3 ) / .7;					
				}
			}
		}


        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.C && bPressed == true )
			{
				if( m_nStatus == enumCloakingStatus.Cloaked || m_nStatus == enumCloakingStatus.Cloaking )
				{
					m_nStatus = enumCloakingStatus.UnCloaking;
				}
				else
				{
					m_nStatus = enumCloakingStatus.Cloaking;
				}
			}
		}


		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 && nYOffset == 0 ) ||
				( nXOffset == 0 && nYOffset == -1 ) ||
				( nXOffset == 1 && nYOffset == 0 ) ||
				( nXOffset == 1 && nYOffset == -1 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}

		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( MGridIsInUse( nXOffset,nYOffset ) == true )
			{
				return( enumModuleType.Other );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}


		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,(int)m_nStatus );
			oSerialize.Set( 2,m_nPercentCloaking );			


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_nStatus = (enumCloakingStatus)oSerialize.GetInt( 1 );
			m_nPercentCloaking = oSerialize.GetDouble( 2 );
		}



		#region properties
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Other );
			}
		}
		#endregion
	}
	public class ScatterLauncher : Module
	{
		#region Properties
		private const int m_cSHOTSPERFIRE = 2;
		private const long m_cMESSAGE_FIREID = 0;
		private const double m_cENERGYTOFIRE = 1.0;

		private bool m_bFiring = false;
		private double m_nTimeSinceLastShot = 30;
		private int m_nNextSeedNumber = 1;
		#endregion


		public ScatterLauncher()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public ScatterLauncher( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 1;
			base.MGridHeight = 2;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "ScatterLauncher";
			this.Texture_Space = "ScatterLauncher";
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);

			if( base.StructurePoints > 0 )
			{
				m_nTimeSinceLastShot += nElapsedTime;

				if( m_bFiring == true )
				{				
					Fire(oSession);
				}
			}
		}
		private void Fire(Session oSession)
		{
			double nLaunchVelocity = 200;
			Vector2 vModuleWorldPtUpperLeftCorner = Vector2.Empty;
			Vector2 vLaunchWorldPt = Vector2.Empty;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;
            Zone oMyZone = null;
			ScatterProjectile oShot = null;
			string sMessage = "";
			Random oRand = new Random( m_nNextSeedNumber );
			m_nNextSeedNumber++;


			if( this.ParentEntity.IsDead == false && this.StructurePoints > 0 && m_nTimeSinceLastShot > TimeBetweenShots() )
			{
                oMyZone = oSession.Zones[this.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];

				if( base.ParentEntity.GetEnergyAvailable() > m_cENERGYTOFIRE )
				{
					base.ParentEntity.TakeEnergy( m_cENERGYTOFIRE );

					for( int i=0 ; i<m_cSHOTSPERFIRE ; i++ )
					{
						oShot = new ScatterProjectile();

						//Add our projectile
						vModuleWorldPtUpperLeftCorner = GetUpperLeftCornerWorldPt();
						vLaunchWorldPt = new Vector2( (float)( vModuleWorldPtUpperLeftCorner.X + ( base.MGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
							vModuleWorldPtUpperLeftCorner.Y + GraphicConstants.m_cGRID_HEIGHT );
						vRotatedLaunchWorldPt = base.ParentEntity.GetRotatedWorldPt( vLaunchWorldPt );

						oShot.Angle = base.ParentEntity.Angle - Math.PI / 2.0 + oRand.NextDouble() * 2 - 1;
						oShot.Vel = new Vector2( (float)( Math.Cos( oShot.Angle ) * nLaunchVelocity + base.ParentEntity.Vel.X ),
							(float)( Math.Sin( oShot.Angle ) * -nLaunchVelocity + base.ParentEntity.Vel.Y ) );
						oShot.Pos = vRotatedLaunchWorldPt;
                        oShot.Location.ZoneID = this.ParentEntity.Location.ZoneID;
						oShot.OwnerSocketID = this.ParentEntity.OwnerSocketID;

						//Add him to the session
						m_nTimeSinceLastShot = 0;
                        oMyZone.AddEntity(oSession, oShot, false);
					}

					//Send our message
					sMessage = (m_nNextSeedNumber-1).ToString();
					NetMsg.Send_EntityMessage( Globals.Inst().GameEngine.DirectPlay,Globals.Inst().GameEngine.DirectPlay.Me,
						this.ParentEntity,m_cMESSAGE_FIREID,sMessage );
				}
			}
			else if( Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1 )
			{
                Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
			}
		}
		private double TimeBetweenShots()
		{
			return( .05 + 1 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) ) ); 
		}

        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.Space &&
				Globals.Inst().OurShip != null && 
				Globals.Inst().OurShip.PKey == this.ParentEntity.PKey )
			{
				m_bFiring = bPressed;
			}
		}
        public override void NetworkMessageReceived(Session oSession, long nMessageType, string sMessage)
		{
			try
			{
				if( nMessageType == m_cMESSAGE_FIREID )
				{
					m_nNextSeedNumber = Convert.ToInt32( sMessage );
					Fire(oSession);
				}
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}
		}

		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset >= 1 && nXOffset <= 2 && nYOffset == 0 ) ||
				( nXOffset == 1 && nYOffset == 0-1 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( nXOffset >= 0 && nXOffset <= 2 && nYOffset == 0 )
			{
				//return( enumModuleType.OpenSpace );
				return( enumModuleType.Weapon );
			}
			else if( nXOffset == 1 && nYOffset == 0-1 ) 
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_bFiring );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_bFiring = oSerialize.GetBool( 1 );
		}


		#region properties
		public bool Firing
		{
			get
			{
				return( m_bFiring );
			}
			set
			{
				m_bFiring = value;
			}
		}
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		public override bool Active
		{
			get
			{
				return(false);
			}
		}

		#endregion
	}
	public class EntrophyShield : Module
	{
		#region Properties
		private double m_nShieldableDamageAmount = 0;
		private bool m_bRenderingShield = false;
		private bool m_bEnergyBillPaid = false;
		private double m_nTimeSinceLastEnergyCheck = m_cMINTIMEBETWEENENERGYCHECK;

		private const double m_cMINTIMEBETWEENENERGYCHECK = 2;
		#endregion


		public EntrophyShield()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public EntrophyShield( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 1;
			base.MGridHeight = 1;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "EntrophyShield";
			this.Texture_Space = "";
		}


        public override void Advance(Session oSession, double nElapsedTime)
		{
			double nEnergyFound = 0;


            base.Advance(oSession, nElapsedTime);

			m_nTimeSinceLastEnergyCheck += nElapsedTime;

			if( base.ModuleIsAvailable() == true &&
				base.ParentEntity != null &&
				base.ParentEntity.GetEnergyAvailable() >= 5*nElapsedTime &&
				(
				m_bEnergyBillPaid == true ||
				m_nTimeSinceLastEnergyCheck >= m_cMINTIMEBETWEENENERGYCHECK 
				)
				)
			{
				m_nTimeSinceLastEnergyCheck = 0;
				m_bEnergyBillPaid = true;
				nEnergyFound = base.ParentEntity.TakeEnergy( 5*nElapsedTime );

				//Raise our damage absorption amount
				m_nShieldableDamageAmount += nElapsedTime * 20;
				m_nShieldableDamageAmount = DSMisc.Min( m_nShieldableDamageAmount,GetMaxShieldableDamageAmount() );
			}
			else
			{
				m_bEnergyBillPaid = false;
				m_nShieldableDamageAmount -= nElapsedTime * 20;
				m_nShieldableDamageAmount = DSMisc.Max( m_nShieldableDamageAmount,0 );
			}
		}

		private double GetMaxShieldableDamageAmount()
		{
			return( 200.0 * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		public override void Render( RenderSettings oRenderSettings,Chassis oParentChassis )
		{
			Vector2 vRelativePos = Vector2.Empty;
			Rectangle oTargetRect = Rectangle.Empty;


			if( oRenderSettings.RenderType == enumRenderType.AboveShip && oRenderSettings.InGameScreen != enumGameScreen.Editor &&
				m_bEnergyBillPaid == true && base.ModuleIsAvailable() == true && m_bRenderingShield == false )
			{
				double nTransPerc = 0;

				m_bRenderingShield = true;
				nTransPerc = .8 - .65 * ( m_nShieldableDamageAmount / GetMaxShieldableDamageAmount() );

				RenderSettings oTempRenderSettings = new RenderSettings( oRenderSettings.AreaToDrawAt,
					oRenderSettings.EntityToCenterOn,oRenderSettings.EntityOffset,
					oRenderSettings.ZoomLevel /*+ .1*/,enumRenderType.Topside,
					enumEditorObject.Modules,enumGameScreen.InSpace,nTransPerc,
					System.Drawing.Color.Blue );
				this.ParentEntity.Render( oTempRenderSettings );
				m_bRenderingShield = false;
			}
		}


		public override double GetPreArmorDamageReduction(double nDamage)
		{
			double nNewDamage = nDamage;

			if ( m_bEnergyBillPaid == true && base.ModuleIsAvailable() == true )
			{
				if( nNewDamage > m_nShieldableDamageAmount )
				{
					nNewDamage -= m_nShieldableDamageAmount;
					m_bEnergyBillPaid = false;
					m_nShieldableDamageAmount = 0;
					this.DealDamage( nNewDamage,null );
				}
				else
				{
					m_nShieldableDamageAmount -= nNewDamage;
					nNewDamage = 0;
				}
			}

			return( nNewDamage );
		}


		public override void Assign()
		{
			m_nShieldableDamageAmount = 0;
			m_bRenderingShield = false;
			m_bEnergyBillPaid = false;
			m_nTimeSinceLastEnergyCheck = m_cMINTIMEBETWEENENERGYCHECK;
		}
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			return( nXOffset == 0 && nYOffset == 0 );
		}

		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( MGridIsInUse( nXOffset,nYOffset ) == true )
			{
				return( enumModuleType.ShieldGen );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}

		public override double GetPropertyTotal(enumEntProperties nPropertyToGet, bool bIncludeChildren)
		{
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );
			double nValue = 0;

			if( nPropertyToGet == enumEntProperties.Shield_Current &&
				//m_bEnergyBillPaid == true && 
				base.ModuleIsAvailable() == true )
			{
				nValue = m_nShieldableDamageAmount;
			}
			else if( nPropertyToGet == enumEntProperties.Shield_Max && base.ModuleIsAvailable() == true )
			{
				nValue = GetMaxShieldableDamageAmount();
			}

			return( nBaseValue + nValue );
		}


		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_bEnergyBillPaid );
			oSerialize.Set( 2,m_nShieldableDamageAmount );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			m_bEnergyBillPaid = oSerialize.GetBool( 1 );
			m_nShieldableDamageAmount = oSerialize.GetDouble( 2 );

			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
		}



		#region Properties
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.HUD );
			}
		}
		#endregion
	}
	public class PointDefensePlasma : Module
	{
		#region Properties
		private const double m_cANGLECHANGERATE = .1;
		private const double m_cENERGYTOFIRE = 1.0;
		private const double m_cMAXFIRERANGE = 1000.0;
		private const double m_cTIMEBETWEENSHOTS = 1.0;

		private double m_nPercDeployed = 0;
		private double m_nTurrentAngle = 0;
		private double m_nTimeSinceLastShot = 30;
		private Entity m_oCurrentTarget = null;
		private double m_nTimeSinceLastEntityCheck = 0;
		#endregion


		public PointDefensePlasma()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}

		public PointDefensePlasma( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 3;
			base.MGridHeight = 3;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "PointDefensePlasma_Launcher_Editor";
			this.Texture_Space = "";
		}


        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);

			if( base.ModuleIsAvailable() == true)
			{
				m_nTimeSinceLastEntityCheck += nElapsedTime;
				m_nTimeSinceLastShot += nElapsedTime;

				if( m_nTimeSinceLastEntityCheck > 2 )
				{
					m_nTimeSinceLastEntityCheck = 0;
					m_oCurrentTarget = LookForTarget(oSession);
				}
                AdvanceTurrent(oSession, m_oCurrentTarget, nElapsedTime);
			}
		}
		private Entity LookForTarget(Session oSession)
		{
			double nClosestDist = 0;
			double nTempDist = 0;
			double nMaxRange = 0;
            Zone oMyZone = null;
			Entity oTargetEntity = null;
            Entity oClosestEntity = null;
			Vector2 vFirePt = Vector2.Empty;

			
			if( base.ModuleIsAvailable() == true )
			{
                oMyZone = oSession.Zones[base.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];

				vFirePt = this.GetRotatedCenterWorldPt();
                foreach( Entity oLoopEntity in oMyZone.Entitys.Values )
				{
					nTempDist = DSMath.Distance( oLoopEntity.Pos,vFirePt );
					nMaxRange = GetMaxFireRange();					
					if( oLoopEntity.EntityType == enumEntityType.Projectile &&
						oLoopEntity.OwnerSocketID != base.ParentEntity.OwnerSocketID &&
						oLoopEntity.IsVisible == true && oLoopEntity.IsInPlay == true &&
						nTempDist < nMaxRange &&
						(
                            oClosestEntity == null || nTempDist < nClosestDist
						)
					  )
					{
						nClosestDist = nTempDist;
                        oClosestEntity = oLoopEntity;
					}
				}

				//Did we find a target?
                if (oClosestEntity != null)
				{
                    oTargetEntity = oClosestEntity;
				}
			}


			return( oTargetEntity );
		}
        private void AdvanceTurrent(Session oSession, Entity oTarget, double nElapsedTime)
		{
			double nAngleOfShip = 0;
			double nDeployFactor = 1.5;
			double nAngleToTarget = 0;
			Vector2 vFirePt = Vector2.Empty;


			//Do we have a target?  If not start to close shop
			if( oTarget == null )
			{
				if( m_nPercDeployed > 0 )
				{
					m_nPercDeployed = DSMisc.Max( 0,m_nPercDeployed-nElapsedTime * nDeployFactor );
				}
			}
				//Ah ha, target.  Deploy if we need to and fire
			else
			{
				//Angle between them and us
				vFirePt = this.GetRotatedCenterWorldPt();
				nAngleOfShip = this.ParentEntity.Angle;// + Math.PI / 2.0;
				nAngleToTarget = DSMath.CalculateRadAngle( vFirePt.X,vFirePt.Y,oTarget.Pos.X,oTarget.Pos.Y );
				nAngleToTarget += nAngleOfShip - Math.PI / 2.0;

				//We have a target, are we deployed?
				if( m_nPercDeployed < 1 )
				{
					m_nPercDeployed = DSMisc.Min( 1,m_nPercDeployed+nElapsedTime * nDeployFactor );
				}
					//Are we facing our target?
				else //if( DSMath.AngleRadDiff( nAngleToTarget,m_nTurrentAngle ) > .1 )
				{
					m_nTurrentAngle = DSMath.MoveRadAngleTwordsRadAngle( m_nTurrentAngle,GetAngleChangeRate(),nAngleToTarget );
				}

				//Can we fire again yet?
				if( m_nPercDeployed == 1 &&
					base.ParentEntity.GetEnergyAvailable() > m_cENERGYTOFIRE &&
					m_nTimeSinceLastShot > TimeBetweenShots() && 
					DSMath.AngleRadDiff( nAngleToTarget,m_nTurrentAngle ) < .05 )
				{
					Fire(oSession);
				}
			}
		}
		private void Fire(Session oSession)
		{
			double nLaunchVelocity = 500;
            Zone oMyZone = null;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;
			UrquanPlasmaProjectile oShot = new UrquanPlasmaProjectile();


			if( this.ModuleIsAvailable() == true && m_nTimeSinceLastShot > TimeBetweenShots() &&
				base.ParentEntity.GetEnergyAvailable() > m_cENERGYTOFIRE )
			{
                oMyZone = oSession.Zones[base.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];
				base.ParentEntity.TakeEnergy( m_cENERGYTOFIRE );
				
				if( Globals.Inst().GameEngine.DirectPlay.Me.SocketID == this.ParentEntity.OwnerSocketID )
				{
					//Add our missile
					vRotatedLaunchWorldPt = GetRotatedCenterWorldPt();

					double nAngleOfShip = -this.ParentEntity.Angle + Math.PI / 2.0;
					oShot.Angle = -( nAngleOfShip + m_nTurrentAngle );
					oShot.Vel = new Vector2( (float)( Math.Cos( oShot.Angle ) * nLaunchVelocity + base.ParentEntity.Vel.X ),
						(float)( Math.Sin( oShot.Angle ) * -nLaunchVelocity + base.ParentEntity.Vel.Y ) );
					oShot.Pos = vRotatedLaunchWorldPt;				
					oShot.OwnerSocketID = this.ParentEntity.OwnerSocketID;
					oShot.ImpactDamage = 1;
					oShot.ExplodeDamage = 0;

					//Add him to the session
					m_nTimeSinceLastShot = 0;
                    oMyZone.AddEntity( oSession,oShot, true);
				}
			}
		}

		private double GetMaxFireRange()
		{
			return( m_cMAXFIRERANGE * ( base.StructurePoints / base.MaxStructurePoints ) ); 
		}
		private double GetAngleChangeRate()
		{
			return( m_cANGLECHANGERATE * ( base.StructurePoints / base.MaxStructurePoints ) ); 
		}
		private double TimeBetweenShots()
		{
			return( m_cTIMEBETWEENSHOTS + 15 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) ) ); 
		}


		public override void Assign()
		{
			m_nPercDeployed = 0;
		}
		public override void UnAssign()
		{
			m_nPercDeployed = 0;
		}


		public override void Render( RenderSettings oRenderSettings,Chassis oParentChassis )
		{
			double nAngleOfShip = this.ParentEntity.Angle;// + Math.PI / 2.0;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
				base.RenderModule3D( GetTextureToUse( oRenderSettings ),
					oRenderSettings,nAngleOfShip - m_nTurrentAngle );
			}
		}

		private string GetTextureToUse( RenderSettings oRenderSettings )
		{
			string sRetVal = "";

			//if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
		{
			if( m_nPercDeployed >= 1.0 )
			{
				if( base.StructurePoints / base.MaxStructurePoints < 0 )
				{
					sRetVal = "PointDefensePlasma_Launcher_9";//_Destroyed";
				}
				else if( base.StructurePoints / base.MaxStructurePoints < .5 )
				{
					sRetVal = "PointDefensePlasma_Launcher_9";//_Damaged";
				}
				else
				{
					sRetVal = "PointDefensePlasma_Launcher_9";
				}
			}
			else
			{
				int nFrame = (int)( ( ( m_nPercDeployed / 10.0 ) * 100 ) );
				sRetVal = "PointDefensePlasma_Launcher_" + nFrame.ToString();
			}
		}

			return( sRetVal );
		}


		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 1 && nYOffset == 0 ) ||
				( nXOffset >= 0 && nXOffset <= 3 && nYOffset == -1 ) ||
				( nXOffset == 1 && nYOffset == -2 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 1 && nYOffset == 0 ) ||
				( nXOffset >= 0 && nXOffset <= 3 && nYOffset == -1 ) ||
				( nXOffset == 1 && nYOffset == -2 ) )
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}



		#region properties
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		#endregion
	}
	public class EtherialFin : Module
	{
		#region Properties
		private const double m_cDRAG_VEL = 80;
		private const double m_cDRAG_ANGULARVEL = 0;

		private const double m_cTURN_VEL = 1500;

		private const double m_cENERGYPERSECOND = 1;
		#endregion


		public EtherialFin()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}

		public EtherialFin( ComplexEntity oParentEntity )
		{
			//base.PKey = DSMisc.GetGUID();
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 1;
			base.MGridHeight = 2;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "EtherialFin";
			this.Texture_Space = "";
		}


        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);

			if( this.ModuleIsAvailable() == true && base.ParentEntity.GetEnergyAvailable() > m_cENERGYPERSECOND * nElapsedTime )
			{
				base.ParentEntity.TakeEnergy( m_cENERGYPERSECOND * nElapsedTime );

				base.ParentEntity.Location = ApplyEtherialDrag( nElapsedTime,base.ParentEntity.Location );
				if( base.ParentEntity.ServerLocation != null )
				{
					base.ParentEntity.ServerLocation = ApplyEtherialDrag( nElapsedTime,base.ParentEntity.ServerLocation );
				}
			}
		}
		private double GetVelTurnAmount()
		{
			return( m_cTURN_VEL * ( base.StructurePoints / base.MaxStructurePoints ) );
		}
		private Location ApplyEtherialDrag( double nElapsedTime,Location oMotion )
		{
			Location oRetLoc = new Location( oMotion );
			double nPercToChange = 0;
			double nAngleOfVel = 0;


			//Slow the ship down by X velocity a second
			//oRetLoc.Vel = new Vector2( 
			//	(float)DSMath.MoveTowords( 0,oMotion.Vel.X,m_cDRAG_VEL*nElapsedTime ),
			//	(float)DSMath.MoveTowords( 0,oMotion.Vel.Y,m_cDRAG_VEL*nElapsedTime ) );
			//oRetLoc.AngularMomentum = DSMath.MoveTowords( 0,oMotion.AngularMomentum,m_cDRAG_ANGULARVEL*nElapsedTime );

			//Now turn the ship to the current direction
			nAngleOfVel = DSMath.CalculateRadAngle( 0,0,oRetLoc.Vel.X,oRetLoc.Vel.Y );
			if( oRetLoc.Vel.Length() > 0 )
			{
				nPercToChange = ( GetVelTurnAmount() * nElapsedTime ) / oRetLoc.Vel.Length();
				nPercToChange = DSMisc.Min( 1,nPercToChange );
				nPercToChange = DSMisc.Max( 0,nPercToChange );
			
				oRetLoc.Vel = new Vector2( 
					(float)( -Math.Cos( oRetLoc.Angle + Math.PI / 2.0 ) * nPercToChange * oRetLoc.Vel.Length() + 
					Math.Cos( nAngleOfVel ) * (1 - nPercToChange) * oRetLoc.Vel.Length() ),
					(float)( -Math.Sin( oRetLoc.Angle + Math.PI / 2.0 ) * nPercToChange * oRetLoc.Vel.Length() + 
					Math.Sin( nAngleOfVel ) * (1 - nPercToChange) * oRetLoc.Vel.Length() ) );
			}

			return( oRetLoc );
		}


		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			double nValue = 0;
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );

			if( nPropertyToGet == enumEntProperties.Movement_EtherialLinearDrag )
			{
				nValue = m_cDRAG_VEL;
			}

			return( nBaseValue + nValue );
		}

		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 && nYOffset == 0 ) ||
				( nXOffset == 0 && nYOffset == 0-1 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}

		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 && nYOffset == 0 ) ||
				( nXOffset == 0 && nYOffset == 0-1 ) )
			{
				return( enumModuleType.Engine );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}



		#region properties
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Engine );
			}
		}
		#endregion
	}
	public class LimpitLauncher : Module
	{
		#region Properties
		private bool m_bFiring = false;
		private double m_nTimeSinceLastShot = 30;

		private double m_nRechargeTime = 2.5;
		private double m_nECostPerShot = 5;
		#endregion


		public LimpitLauncher()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public LimpitLauncher( ComplexEntity oParentEntity )
		{
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 2;
			base.MGridHeight = 2;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "Limpit_Launcher";
		}
		public override void CreatedAsNewItem()
		{
			base.CreatedAsNewItem();

			if( this.ItemLevel == enumItemLevel.Normal )
			{
				m_nRechargeTime = DSMisc.GetRnd() * .4 - .2 + .5;
				m_nECostPerShot = DSMisc.GetRnd() * 10.0 - 5.0 + 20.0;
			}
			else if( this.ItemLevel == enumItemLevel.Advanced )
			{
				m_nRechargeTime = DSMisc.GetRnd() * .4 - .2 + .3;
				m_nECostPerShot = DSMisc.GetRnd() * 20.0 - 10.0 + 60.0;
			}
			else if( this.ItemLevel == enumItemLevel.Expert )
			{
				m_nRechargeTime = DSMisc.GetRnd() * .4 - .2 + .1;
				m_nECostPerShot = DSMisc.GetRnd() * 50.0 - 25.0 + 120.0;
			}			
		}

		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();

			oProperties.Add( "Damage: N/A" );
            if (TimeBetweenShots() != m_nRechargeTime)
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " (" + m_nRechargeTime.ToString("0.0") + ") seconds");
            }
            else
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " seconds");
            }	
			oProperties.Add( "Costs " + m_nECostPerShot.ToString( "0.0" ) + " per shot" );
			
			return( oProperties );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);
			
			if( base.StructurePoints > 0 )
			{
				m_nTimeSinceLastShot += nElapsedTime;

				if( m_bFiring == true && m_nTimeSinceLastShot > TimeBetweenShots() )
				{				
					Fire(oSession);
				}
			}
		}
		public void Fire(Session oSession)
		{
			double nLaunchVelocity = 0;
            Zone oMyZone = null;
			Vector2 vModuleWorldPtUpperLeftCorner = Vector2.Empty;
			Vector2 vLaunchWorldPt = Vector2.Empty;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;
			LimpitProjectile oShot = new LimpitProjectile();


            if (this.ModuleIsAvailable() == true && this.CanLaunchProjectile() == true && m_nTimeSinceLastShot > TimeBetweenShots())
			{
				if( base.ParentEntity.GetEnergyAvailable() > m_nECostPerShot )
				{
                    oMyZone = oSession.Zones[base.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];
					base.ParentEntity.TakeEnergy( m_nECostPerShot );
					 
					//Add our missile
					vModuleWorldPtUpperLeftCorner = GetUpperLeftCornerWorldPt();
					vLaunchWorldPt = new Vector2( 
						(float)( vModuleWorldPtUpperLeftCorner.X + ( base.MGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
						(float)( vModuleWorldPtUpperLeftCorner.Y /*+ ( base.MGridHeight * .25 ) * GraphicConstants.m_cGRID_HEIGHT*/ ) );
					vRotatedLaunchWorldPt = base.ParentEntity.GetRotatedWorldPt( vLaunchWorldPt );

					oShot.Angle = base.ParentEntity.Angle - Math.PI / 2.0;
					oShot.Vel = new Vector2( (float)( Math.Cos( oShot.Angle ) * nLaunchVelocity + base.ParentEntity.Vel.X ),
						(float)( Math.Sin( oShot.Angle ) * nLaunchVelocity + base.ParentEntity.Vel.Y ) );
					oShot.Pos = vRotatedLaunchWorldPt;
                    oShot.Location.ZoneID = this.ParentEntity.Location.ZoneID;
					oShot.OwnerSocketID = this.ParentEntity.OwnerSocketID;

					//Add him to the session
					m_nTimeSinceLastShot = 0;
                    oMyZone.AddEntity( oSession,oShot, true);
				}
				else if( Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1 )
				{
                    Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
				}
			}
		}

		private double TimeBetweenShots()
		{
			double nBaseRate = 0;
			double nMod = 0;

            nBaseRate = this.GetPropertyTotal(enumEntProperties.Weapon_SecondsToRecharge, false);
            nMod = 1 + this.GetPropertyTotal(enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt, false);
            if (this.Chassis != null)
            {
                nBaseRate += this.Chassis.GetPropertyTotal(enumEntProperties.Weapon_SecondsToRecharge, false);
                nMod += this.Chassis.GetPropertyTotal(enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt, false);
            }
			nBaseRate = nBaseRate * nMod;
 
			return( nBaseRate ); 
		}

        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.Space &&
				Globals.Inst().OurShip != null && 
				Globals.Inst().OurShip.PKey == this.ParentEntity.PKey )
			{
				if( bPressed == true )
				{
					Fire(oSession);
				}
				m_bFiring = bPressed;
			}
		}
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( nXOffset >= 0 && nXOffset <= 1 &&
				nYOffset <= 0 && nYOffset >= -1 )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( MGridIsInUse( nXOffset,nYOffset ) == true )
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}
		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );
			double nMyValue = 0;

			if( nPropertyToGet == enumEntProperties.Weapon_SecondsToRecharge )
			{
				nMyValue = m_nRechargeTime + 15 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) );
			}

			return( nBaseValue + nMyValue );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nRechargeTime );
			oSerialize.Set( 2,m_nECostPerShot );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_nRechargeTime = oSerialize.GetDouble( 1 );
			m_nECostPerShot = oSerialize.GetDouble( 2 );
		}


		#region properties
		public bool Firing
		{
			get
			{
				return( m_bFiring );
			}
			set
			{
				m_bFiring = value;
			}
		}
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		public override string Name
		{
			get
			{
				string sName = "";

				if( this.ItemLevel == enumItemLevel.Normal )
				{
					sName = "Limpit Launcher";
				}
				else if( this.ItemLevel == enumItemLevel.Advanced )
				{
					sName = "Adv. Limpit Launcher";
				}
				else if( this.ItemLevel == enumItemLevel.Expert )
				{
					sName = "Limpit Spawner";
				}			

				return( sName );
			}
		}
		#endregion
	}
	public class VuxLazer : Module
	{
		#region Properties
		private const string m_cLAZER_FIRING_SOUNDKEY = "Vux_Lazer";
		private const string m_cLAZER_HITTING_SOUNDKEY = "Vux_LazerHit";		
		private const double m_cLAZERONTIME = .339;
		private const double m_cLAZERLEN = 128 * 3.0;

		private double m_nTimeLazerTurnedOn = 0;
		private bool m_bLazerOn = false;
		private bool m_bFiring = false;
		private double m_nTimeSinceLastShot = 30;
		private double m_nTimeSinceLastDamageCheck = 30;
		private double m_nMinDamage = 0;
		private double m_nMaxDamage = 0;
		private double m_nECostPerShot = 5;

		private double m_nLazerDistance = 0;
		private int m_nLazerFiringSoundIndex = -1;
		private Entity m_oLazerCurrentTarget = null;
		#endregion


		public VuxLazer()
		{
			//Do not put code here
			throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public VuxLazer( ComplexEntity oParentEntity )
		{
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 3;
			base.MGridHeight = 5;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "Vux_Lazer";
		}
		public override void CreatedAsNewItem()
		{
			base.CreatedAsNewItem();

			if( this.ItemLevel == enumItemLevel.Normal )
			{
				m_nMinDamage = 1 + DSMisc.GetRnd() * 10.0 - 5.0;
				m_nMaxDamage = 5 + DSMisc.GetRnd() * 10.0 - 5.0;
				m_nECostPerShot = DSMisc.GetRnd() * 10.0 - 5 + 45.0;
			}
			else if( this.ItemLevel == enumItemLevel.Advanced )
			{
				m_nMinDamage = 15 + DSMisc.GetRnd() * 30.0 - 15.0;
				m_nMaxDamage = 20 + DSMisc.GetRnd() * 30.0 - 15.0;
				m_nECostPerShot = DSMisc.GetRnd() * 30.0 - 15.0 + 100.0;
                this.MinLevel = DSMisc.GetRnd(4, 6);
			}
			else if( this.ItemLevel == enumItemLevel.Expert )
			{
				m_nMinDamage = 35 + DSMisc.GetRnd() * 50.0 - 25.0;
				m_nMaxDamage = 60 + DSMisc.GetRnd() * 50.0 - 25.0;
				m_nECostPerShot = DSMisc.GetRnd() * 50.0 - 25.0 + 210.0;
                this.MinLevel = DSMisc.GetRnd(8, 10);
			}			
		}

		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();

			oProperties.Add( "Damage: " + m_nMinDamage.ToString( "0" ) + " - " + m_nMaxDamage.ToString( "0" ) );
			oProperties.Add( "Costs " + m_nECostPerShot.ToString( "0.0" ) + " per shot" );

			return( oProperties );
		}
		public override void Render( RenderSettings oRenderSettings,Chassis oParentChassis )
		{
			LoadedTexture oLoadedTexture = null;
			Vector2 vRelativePos = Vector2.Empty;
			Rectangle oTargetRect = Rectangle.Empty;
			Rectangle oRenderRect = Rectangle.Empty;
			string sTextureKey = "Vux_Beam";


			if( oRenderSettings.RenderType == enumRenderType.AboveShip && oRenderSettings.InGameScreen != enumGameScreen.Editor &&
				base.ModuleIsAvailable() == true && m_bLazerOn == true )
			{
				oLoadedTexture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( sTextureKey );

				//Find the beam origin point
				Vector2 vBeamLoc = this.GetUpperLeftCornerWorldPt();
				vBeamLoc.X += (float)( GraphicConstants.m_cGRID_WIDTH * 1.5 - oLoadedTexture.Size.X / 2.0 );
				vBeamLoc.Y -= (float)LazerLen();
				Vector2 vBeamScreenLoc = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vBeamLoc );

				//The rotate around point is the center of the ships center of mass
				Vector2 vShipCenter = this.ParentEntity.BaseChassis.GetCenterWorldPt();
				Vector2 vShipCenterScreenLoc = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vShipCenter );
				Vector2 vRotateAround = vShipCenterScreenLoc - vBeamScreenLoc;

				oRenderRect = new System.Drawing.Rectangle( 
					(int)vBeamScreenLoc.X,(int)vBeamScreenLoc.Y,
					(int)( oLoadedTexture.Size.X * oRenderSettings.ZoomLevel ),
					(int)( LazerLen() * oRenderSettings.ZoomLevel ) );
				Globals.Inst().GameEngine.RenderTexture2D( sTextureKey,
					System.Drawing.Rectangle.Empty,oRenderRect,
					vRotateAround,this.ParentEntity.Angle,oRenderSettings.PercentTransparent,false,
					oRenderSettings.BaseDrawColor.ToArgb() );
			}
		}

		public override void Advance(Session oSession,double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);
			
			if( base.ModuleIsAvailable() == true )
			{
				if( m_bFiring == false || Globals.Inst().GameEngine.AppTime - m_nTimeLazerTurnedOn > m_cLAZERONTIME )
				{
					m_bLazerOn = false;
				}

				m_nTimeSinceLastShot += nElapsedTime;
				m_nTimeSinceLastDamageCheck += nElapsedTime;

                if (m_bFiring == true || m_bLazerOn == true)
                {
                    if (base.ParentEntity.GetEnergyAvailable() > m_nECostPerShot * nElapsedTime)
                    {
                        base.ParentEntity.TakeEnergy(m_nECostPerShot * nElapsedTime);

                        if (m_bFiring == true && m_bLazerOn == false)
                        {
                            ActivateLazer(nElapsedTime);
                        }
                    }
                    else if (Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1)
                    {
                        Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
                        m_bLazerOn = false;
                    }
                }

				CheckForLazerDamage(oSession, nElapsedTime );
			}
			else 
			{
				m_bLazerOn = false;
			}

			AdvanceSound();
		}
		public void StartFiring( double nElapsedTime )
		{
			m_bFiring = true;
		}
		public void StopFiring( double nElapsedTime )
		{
			m_bFiring = false;
		}
		private void ActivateLazer( double nElapsedTime )
		{
			if( m_bLazerOn == false && base.ModuleIsAvailable() == true && m_nTimeSinceLastShot > TimeBetweenShots() )
			{
/*				if( base.ParentEntity.GetEnergyAvailable() > m_nECostPerShot )
				{
					base.ParentEntity.TakeEnergy( m_nECostPerShot );*/
					m_bLazerOn = true;					
					m_nTimeLazerTurnedOn = Globals.Inst().GameEngine.AppTime;
				/*}
				else if( Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1 )
				{
					Globals.Inst().GameEngine.DirectSound.PlaySound( "NoEnergy" );
				}*/
			}
			else
			{
				m_bLazerOn = false;
			}
		}

		private void CheckForLazerDamage( Session oSession,double nElapsedTime )
		{
			Entity oMinEntity = null;
			RegionLine oLazer = null;
			RegionPoint oTempHitPoint = null;
			RegionPoint oMinHitPoint = null;
            Zone oMyZone = null;
			double nDamage = 0;
			double nMinDist = -1;
			double nMAngle = 0;
			double nTempMinDist = 0;


			if( m_bLazerOn == true )
            {
                oMyZone = oSession.Zones[this.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];

                //Find the beam origin point
				Vector2 vBeamLoc = this.GetUpperLeftCornerWorldPt();
				vBeamLoc.X += (float)( GraphicConstants.m_cGRID_WIDTH * 1.5 );

				nMAngle = Globals.ShipAngleToMathAngle( this.ParentEntity.Angle );
				Vector2 vBeamStart = this.ParentEntity.GetRotatedWorldPt( vBeamLoc );
				Vector2 vBeamStop = new Vector2( (float)( vBeamStart.X + Math.Cos( nMAngle ) * m_cLAZERLEN ),
												 (float)( vBeamStart.Y - Math.Sin( nMAngle ) * m_cLAZERLEN ) );
				oLazer = new RegionLine( vBeamStart,vBeamStop ); 


				if( m_nTimeSinceLastDamageCheck > .25 )
				{
					m_nTimeSinceLastDamageCheck = 0;

					//Look for a target
                    foreach( Entity oLoopEntity in oMyZone.Entitys.Values )
					{
						if( oLoopEntity != this.ParentEntity && oLoopEntity.EntityType != enumEntityType.Loot &&
							oLoopEntity.OwnerSocketID != this.ParentEntity.OwnerSocketID &&
							DSMath.Distance( oLoopEntity.Pos,this.ParentEntity.Pos ) < 1000 )
						{
							nTempMinDist = CalculateLazLengthTarget( oLazer,oLoopEntity,ref oTempHitPoint );
							if( nTempMinDist >= 0 && ( oMinHitPoint == null || nTempMinDist < nMinDist ) )
							{
								oMinHitPoint = oTempHitPoint;
								nMinDist = nTempMinDist;
								oMinEntity = oLoopEntity;
							}
						}
					}
				}
				else if( m_oLazerCurrentTarget != null )
				{
					nMinDist = CalculateLazLengthTarget( oLazer,m_oLazerCurrentTarget,ref oMinHitPoint );
					if( nMinDist != -1 )
					{
						oMinEntity = m_oLazerCurrentTarget;
					}
				}

				//Ok, did we find a target?
				if( oMinEntity != null )
				{
					m_oLazerCurrentTarget = oMinEntity;
					m_nLazerDistance = nMinDist + .1;
					nDamage = DSMisc.GetRnd() * ( m_nMaxDamage - m_nMinDamage ) + m_nMinDamage;
                    oMinEntity.DealDamage(oSession, this.ParentEntity, nDamage, oMinHitPoint);
                    Globals.PlaySound(m_cLAZER_HITTING_SOUNDKEY, this.ParentEntity.Location);
				}
				else
				{
					m_oLazerCurrentTarget = null;
					m_nLazerDistance = 0;
				}
			}
		}
		private double CalculateLazLengthTarget( RegionLine oLazer,Entity oTarget,ref RegionPoint oMinHitPointPoint )
		{
			double nMinDist = -1;
			double nTempMinDist = 0;
			RegionPoint oTempHitPoint = null;
			RegionPoint oMinHitPoint = null;
			ArrayList oRegions = null;


			oRegions = oTarget.GetRegions();
			for( int nRgsIdx=0 ; nRgsIdx<oRegions.Count ; nRgsIdx++ )
			{
				oTempHitPoint = oLazer.HitTest( (Region)oRegions[nRgsIdx] );

				if( oTempHitPoint != null )
				{
					nTempMinDist = DSMath.Distance( oTempHitPoint.Center,oLazer.StartPt );
					if( oMinHitPoint == null || nTempMinDist < nMinDist )
					{
						oMinHitPointPoint = oTempHitPoint;
						nMinDist = nTempMinDist;
					}
				}
			}


			return( nMinDist );
		}
		private void AdvanceSound()
		{
			if( Globals.Inst().IAmTheServer == false )
			{
				if( m_bLazerOn == true )
				{
					if( m_nLazerFiringSoundIndex == -1 || 
						Globals.Inst().GameEngine.DirectSound.SoundIsStillPlaying( m_cLAZER_FIRING_SOUNDKEY,m_nLazerFiringSoundIndex ) == false )
					{
                        m_nLazerFiringSoundIndex = Globals.PlaySound(m_cLAZER_FIRING_SOUNDKEY, this.ParentEntity.Location);
					}				
				}
				else if( m_nLazerFiringSoundIndex != -1 )
				{
					Globals.Inst().GameEngine.DirectSound.StopPlayingSound( m_cLAZER_FIRING_SOUNDKEY,m_nLazerFiringSoundIndex );
				}
			}
		}
		private double TimeBetweenShots()
		{
			double nBaseRate = 0;
			double nMod = 0;

			nBaseRate = this.Chassis.GetPropertyTotal( enumEntProperties.Weapon_SecondsToRecharge,false ) + 
				this.GetPropertyTotal( enumEntProperties.Weapon_SecondsToRecharge,false );
			nMod = 1 + this.Chassis.GetPropertyTotal( enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt,false ) +
				this.GetPropertyTotal( enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt,false );
			nBaseRate = nBaseRate * nMod;
 
			return( nBaseRate ); 
		}
		private double LazerLen()
		{
			if( m_nLazerDistance == 0 )
			{
				return( m_cLAZERLEN );
			}
			else
			{
				return( m_nLazerDistance );
			}
		}

        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == Key.Space )
			{
				m_bFiring = bPressed;
			}
		}
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( nXOffset >= 0 && nXOffset <= 2 &&
				nYOffset <= 0 && nYOffset >= -4 )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( nYOffset == 0 )
			{
				return( enumModuleType.OpenSpace );
			}
			else if( MGridIsInUse( nXOffset,nYOffset ) == true )
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}
		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );
			double nMyValue = 0;

			if( nPropertyToGet == enumEntProperties.Weapon_SecondsToRecharge )
			{
				nMyValue = 5 * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) );
			}

			return( nBaseValue + nMyValue );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nECostPerShot );
			oSerialize.Set( 2,m_nMaxDamage );
			oSerialize.Set( 3,m_nMinDamage );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_nECostPerShot = oSerialize.GetDouble( 1 );
			m_nMaxDamage = oSerialize.GetDouble( 2 );
			m_nMinDamage = oSerialize.GetDouble( 3 );
		}


		#region properties
		public bool Firing
		{
			get
			{
				return( m_bFiring );
			}
			set
			{
				m_bFiring = value;
			}
		}
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		public override string Name
		{
			get
			{
				string sName = "";

				if( this.ItemLevel == enumItemLevel.Normal )
				{
					sName = "Lazer";
				}
				else if( this.ItemLevel == enumItemLevel.Advanced )
				{
					sName = "Particle Lazer";
				}
				else if( this.ItemLevel == enumItemLevel.Expert )
				{
					sName = "Mason Lazer";
				}			

				return( sName );
			}
		}
		#endregion
	}
    
    public class ProjectileLauncher : Module
	{
		#region Properties
		private bool m_bFiring = false;
		private double m_nTimeSinceLastShot = 30;

		private Type m_oProjectileType = null;
		private double m_nLaunchAngle = 0;
		private double m_nLaunchVel = 0;

		private double m_nExplodingDamage = 0;
		private double m_nMinDamage = 0;
		private double m_nMaxDamage = 0;
		private double m_nRechargeTime = .5;
		private double m_nRechargeTimeDamagePenalty = 15;
		private double m_nECostPerShot = 20;
		private Key m_oLaunchKey = Key.Space;

		private int m_nMaximumShotsOnScreen = 0;
		private ArrayList m_oShotsOnScreen = new ArrayList();
		#endregion


		public ProjectileLauncher()
		{
			//Do not put code here
			//throw new System.Exception( "Initializing module with wrong constructor." );
		}
		public override ArrayList GetHoverRenderText()
		{
			ArrayList oProperties = base.GetHoverRenderText();

			oProperties.Add( "Damage: " + m_nMinDamage.ToString( "0" ) + " - " + m_nMaxDamage.ToString( "0" ) );
            if (TimeBetweenShots() != m_nRechargeTime)
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " (" + m_nRechargeTime.ToString("0.0") + ") seconds");
            }
            else
            {
                oProperties.Add("Fires every " + TimeBetweenShots().ToString("0.0") + " seconds");
            }	
			oProperties.Add( "Costs " + m_nECostPerShot.ToString( "0.0" ) + " per shot" );

			return( oProperties );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);
			
			if( base.StructurePoints > 0 )
			{
				m_nTimeSinceLastShot += nElapsedTime;

				if( m_bFiring == true )
				{				
					Fire(oSession);
				}
			}
		}
		public void Fire(Session oSession)
		{
			Vector2 vModuleWorldPtUpperLeftCorner = Vector2.Empty;
			Vector2 vLaunchWorldPt = Vector2.Empty;
			Vector2 vRotatedLaunchWorldPt = Vector2.Empty;
			Projectile oShot = null;
            Zone oMyZone = null;
			int nShotsInWorld = 0;


            oMyZone = oSession.Zones[this.ParentEntity.MostUpToDateLoc.ZoneID.ToString()];

			#region Count shots alive
			for( int i=0 ; i<m_oShotsOnScreen.Count ; i++ )
			{
				if( i >= m_oShotsOnScreen.Count ){ break; }

				oShot = (Projectile)m_oShotsOnScreen[ i ];
                if (oShot.IsDead == true || oMyZone.Entitys.ContainsKey(oShot.PKey) == false)
				{
					m_oShotsOnScreen.RemoveAt( i );
					i--;
				}
				else
				{
					nShotsInWorld++;
				}
			}
			#endregion

			if( this.ParentEntity.IsDead == false && this.StructurePoints > 0 && 
				m_nTimeSinceLastShot > TimeBetweenShots() &&
				( m_nMaximumShotsOnScreen == 0 || nShotsInWorld < m_nMaximumShotsOnScreen ) )
			{
				if( base.ParentEntity.GetEnergyAvailable() > m_nECostPerShot )
				{
					base.ParentEntity.TakeEnergy( m_nECostPerShot );
				
				    oShot = (Projectile)Activator.CreateInstance( m_oProjectileType,new object[]{} );
	 
					//Add our missile
					vModuleWorldPtUpperLeftCorner = GetUpperLeftCornerWorldPt();
					vLaunchWorldPt = new Vector2( 
						(float)( vModuleWorldPtUpperLeftCorner.X + ( base.MGridWidth / 2.0 ) * GraphicConstants.m_cGRID_WIDTH ),
						(float)( vModuleWorldPtUpperLeftCorner.Y /*+ ( base.MGridHeight * .25 ) * GraphicConstants.m_cGRID_HEIGHT*/ ) );
					vRotatedLaunchWorldPt = base.ParentEntity.GetRotatedWorldPt( vLaunchWorldPt );
					oShot.Angle = base.ParentEntity.Angle - Math.PI / 2.0 + m_nLaunchAngle;
					oShot.Vel = new Vector2( (float)( Math.Cos( oShot.Angle ) * m_nLaunchVel + base.ParentEntity.Vel.X ),
											 (float)( Math.Sin( oShot.Angle ) * m_nLaunchVel + base.ParentEntity.Vel.Y ) );
					oShot.Pos = vRotatedLaunchWorldPt;
                    oShot.Location.ZoneID = this.ParentEntity.Location.ZoneID;
					oShot.OwnerSocketID = this.ParentEntity.OwnerSocketID;
					oShot.ImpactDamageMin = m_nMinDamage;
					oShot.ImpactDamageMax = m_nMaxDamage;
					oShot.ExplodingDamage = m_nExplodingDamage;

					//Add him to the session
					m_nTimeSinceLastShot = 0;
                    oMyZone.AddEntity( oSession,oShot, true);

					//Are we tracking number of shots alive?
					if( m_nMaximumShotsOnScreen != 0 )
					{
						m_oShotsOnScreen.Add( oShot );
					}
				}
				else if( Globals.Inst().IAmTheServer == false && this.ParentEntity.OwnerSocketID != -1 )
				{
                    Globals.PlaySound("NoEnergy", this.ParentEntity.Location);
				}
			}
		}

		private double TimeBetweenShots()
		{
			double nBaseRate = 0;
			double nMod = 0;

			if( this.Chassis != null )
			{
				nBaseRate = this.Chassis.GetPropertyTotal( enumEntProperties.Weapon_SecondsToRecharge,false );
				nMod = this.Chassis.GetPropertyTotal( enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt,false );
			}
			nBaseRate += this.GetPropertyTotal( enumEntProperties.Weapon_SecondsToRecharge,false );
			nMod += 1 + this.GetPropertyTotal( enumEntProperties.SpecialTraits_Wpn_RechargeRate_Prcnt,false );
			nBaseRate = nBaseRate * nMod;
 
			return( nBaseRate ); 
		}

        public override void KeyProcess(Session oSession, Key oKey, bool bPressed)
		{
            base.KeyProcess(oSession, oKey, bPressed);

			if( oKey == m_oLaunchKey &&
				Globals.Inst().OurShip != null && 
				Globals.Inst().OurShip.PKey == this.ParentEntity.PKey )
			{
				if( bPressed == true )
				{
					Fire(oSession);
				}
				m_bFiring = bPressed;
			}
		}
		public override double GetPropertyTotal( enumEntProperties nPropertyToGet,bool bIncludeChildren )
		{
			double nBaseValue = base.GetPropertyTotal( nPropertyToGet,bIncludeChildren );
			double nMyValue = 0;

			if( nPropertyToGet == enumEntProperties.Weapon_SecondsToRecharge )
			{
				nMyValue = m_nRechargeTime + m_nRechargeTimeDamagePenalty * ( 1 - ( base.StructurePoints / base.MaxStructurePoints ) );
			}

			return( nBaseValue + nMyValue );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_bFiring );
			oSerialize.Set( 2,m_nMinDamage );
			oSerialize.Set( 3,m_nMaxDamage );
			oSerialize.Set( 4,m_nExplodingDamage );
			oSerialize.Set( 5,m_nRechargeTime );
			oSerialize.Set( 6,m_nRechargeTimeDamagePenalty );
			oSerialize.Set( 7,m_nECostPerShot );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_bFiring = false;//oSerialize.GetBool( 1 );

			m_nMinDamage = oSerialize.GetDouble( 2 );
			m_nMaxDamage = oSerialize.GetDouble( 3 );
			m_nExplodingDamage = oSerialize.GetDouble( 4 );
			m_nRechargeTime = oSerialize.GetDouble( 5 );
			m_nRechargeTimeDamagePenalty = oSerialize.GetDouble( 6 );
			m_nECostPerShot = oSerialize.GetDouble( 7 );
		}


		#region properties
		public bool Firing
		{
			get
			{
				return( m_bFiring );
			}
			set
			{
				m_bFiring = value;
			}
		}
		public override enumModuleType ModuleType
		{
			get
			{
				return( enumModuleType.Weapon );
			}
		}
		public double MinDamage
		{
			get
			{
				return( m_nMinDamage );
			}
			set
			{
				m_nMinDamage = value;
			}
		}
		public double MaxDamage
		{
			get
			{
				return( m_nMaxDamage );
			}
			set
			{
				m_nMaxDamage = value;
			}
		}
		public double ExplodingDamage
		{
			get
			{
				return( m_nExplodingDamage );
			}
			set
			{
				m_nExplodingDamage = value;
			}
		}
		public double RechargeTime
		{
			get
			{
				return( m_nRechargeTime );
			}
			set
			{
				m_nRechargeTime = value;
			}
		}
		public double RechargeTimeDamagePenalty
		{
			get
			{
				return( m_nRechargeTimeDamagePenalty );
			}
			set
			{
				m_nRechargeTimeDamagePenalty = value;
			}
		}
		public double ECostPerShot
		{
			get
			{
				return( m_nECostPerShot );
			}
			set
			{
				m_nECostPerShot = value;
			}
		}
		public Type ProjectileType
		{
			get
			{
				return( m_oProjectileType );
			}
			set
			{
				m_oProjectileType = value;
			}
		}
		public double LaunchAngle
		{
			get
			{
				return( m_nLaunchAngle );
			}
			set
			{
				m_nLaunchAngle = value;
			}
		}
		public double LaunchVel
		{
			get
			{
				return( m_nLaunchVel );
			}
			set
			{
				m_nLaunchVel = value;
			}
		}		
		public int MaximumShotsOnScreen
		{
			get
			{
				return( m_nMaximumShotsOnScreen );
			}
			set
			{
				m_nMaximumShotsOnScreen = value;
			}
		}
		public Key LaunchKey
		{
			get
			{
				return( m_oLaunchKey );
			}
			set
			{
				m_oLaunchKey = value;
			}
		}
		#endregion
	}
	public class SpathiPlasmaPelletLauncher : ProjectileLauncher
	{
		#region Properties
		#endregion 

		public SpathiPlasmaPelletLauncher( ComplexEntity oParentEntity )
		{
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 2;
			base.MGridHeight = 2;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "Spathi_Pellet_Gun";
			base.ProjectileType = typeof( PlasmaPellitProjectile );
			base.LaunchVel = 400;
			base.RechargeTime = .09;
			base.MaximumShotsOnScreen = 6;
		}
		public override void CreatedAsNewItem()
		{
			base.CreatedAsNewItem();

			if( this.ItemLevel == enumItemLevel.Normal )
			{
				this.MinDamage = DSMisc.GetRnd( 1,2 ) + 2;
				this.MaxDamage = this.MinDamage + DSMisc.GetRnd( 4,5 );
				//this.RechargeTime = DSMisc.GetRnd() * .4 - .2 + .5;
				this.ECostPerShot = DSMisc.GetRnd() * 1.0 - .5 + 1.0;
			}
			else if( this.ItemLevel == enumItemLevel.Advanced )
			{
				this.MinDamage = DSMisc.GetRnd( 4,5 ) + 8;
				this.MaxDamage = this.MinDamage + DSMisc.GetRnd( 8,12 );
				//this.RechargeTime = DSMisc.GetRnd() * .4 - .2 + .3;
				this.ECostPerShot = DSMisc.GetRnd() * 2.0 - 1.0 + 3.0;
			}
			else if( this.ItemLevel == enumItemLevel.Expert )
			{
				this.MinDamage = DSMisc.GetRnd( 20,30 ) + 10;
				this.MaxDamage = this.MinDamage + DSMisc.GetRnd( 35,40 );
				//this.RechargeTime = DSMisc.GetRnd() * .4 - .2 + .1;
				this.ECostPerShot = DSMisc.GetRnd() * 6.0 - 3.0 + 8.0;
			}			
		}
        
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 || nXOffset == 1 ) && ( nYOffset == 0 || nYOffset == -1 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 || nXOffset == 1 ) && nYOffset == 0 )
			{
				return( enumModuleType.OpenSpace );
			}
			else if( ( nXOffset == 0 || nXOffset == 1 ) && nYOffset == -1 ) 
			{
				return( enumModuleType.Weapon );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}
        
		#region properties
		public override string Name
		{
			get
			{
				string sName = "";

				if( this.ItemLevel == enumItemLevel.Normal )
				{
					sName = "Plasma Shard Launcher";
				}
				else if( this.ItemLevel == enumItemLevel.Advanced )
				{
					sName = "Plasma Pellet Launcher";
				}
				else if( this.ItemLevel == enumItemLevel.Expert )
				{
					sName = "Plasma Ball Launcher";
				}			

				return( sName );
			}
		}
		#endregion
	}
	public class SpathiBUTTLauncher : ProjectileLauncher
	{
		#region Properties
		#endregion 

		public SpathiBUTTLauncher( ComplexEntity oParentEntity )
		{
			base.ParentEntity = oParentEntity;
			base.MGridWidth = 2;
			base.MGridHeight = 2;
			base.StructurePoints = 10;
			base.MaxStructurePoints = 10;
			base.PercentDamageRedux = 0;
			base.FixedDamageRedux = 0;
			this.Texture_Editor = "Spathi_BUTT_Launcher";
			base.ProjectileType = typeof( BUTTProjectile );
			base.LaunchVel = 200;
			base.LaunchAngle = Math.PI;
			base.LaunchKey = Key.M;
		}
		public override void CreatedAsNewItem()
		{
			base.CreatedAsNewItem();

			if( this.ItemLevel == enumItemLevel.Normal )
			{
				this.MinDamage = DSMisc.GetRnd( 30,38 );
				this.MaxDamage = this.MinDamage + DSMisc.GetRnd( 14,18 );
				this.RechargeTime = DSMisc.GetRnd() * .4 - .2 + .8;
				this.ECostPerShot = DSMisc.GetRnd() * 4.0 - 2.0 + 4.0;
			}
			else if( this.ItemLevel == enumItemLevel.Advanced )
			{
				this.MinDamage = DSMisc.GetRnd( 38,40 );
				this.MaxDamage = this.MinDamage + DSMisc.GetRnd( 22,30 );
				this.RechargeTime = DSMisc.GetRnd() * .4 - .2 + .6;
				this.ECostPerShot = DSMisc.GetRnd() * 6.0 - 3.0 + 12.0;
			}
			else if( this.ItemLevel == enumItemLevel.Expert )
			{
				this.MinDamage = DSMisc.GetRnd( 90,100 );
				this.MaxDamage = this.MinDamage + DSMisc.GetRnd( 35,40 );
				this.RechargeTime = DSMisc.GetRnd() * .4 - .2 + .4;
				this.ECostPerShot = DSMisc.GetRnd() * 8.0 - 4.0 + 20.0;
			}			
		}
        
		public override bool MGridIsInUse( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 || nXOffset == 1 ) && ( nYOffset == 0 || nYOffset == -1 ) )
			{
				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override enumModuleType MGridTypeNeeded( int nXOffset,int nYOffset )
		{
			if( ( nXOffset == 0 || nXOffset == 1 ) && nYOffset == 0 )
			{
				return( enumModuleType.All );//Weapon
			}
			else if( ( nXOffset == 0 || nXOffset == 1 ) && nYOffset == -1 ) 
			{
				return( enumModuleType.OpenSpace );
			}
			else
			{
				return( enumModuleType.CantBeUsed );
			}
		}
        
		#region properties
		public override string Name
		{
			get
			{
				string sName = "";

				if( this.ItemLevel == enumItemLevel.Normal )
				{
					sName = "B.U.T.T. Platform";
				}
				else if( this.ItemLevel == enumItemLevel.Advanced )
				{
					sName = "B.U.T.T. Emitter";
				}
				else if( this.ItemLevel == enumItemLevel.Expert )
				{
					sName = "B.U.T.T. Launcher";
				}			

				return( sName );
			}
		}
		#endregion
	}
}
